import { Observable } from 'rxjs';
import { share } from 'rxjs/operators';
import * as io from 'socket.io-client';
export class WrappedSocket {
    constructor(config) {
        this.config = config;
        this.subscribersCounter = {};
        this.eventObservables$ = {};
        this.emptyConfig = {
            url: '',
            options: {}
        };
        if (config === undefined) {
            config = this.emptyConfig;
        }
        const url = config.url;
        const options = config.options;
        const ioFunc = io.default ? io.default : io;
        this.ioSocket = ioFunc(url, options);
    }
    of(namespace) {
        this.ioSocket.of(namespace);
    }
    on(eventName, callback) {
        this.ioSocket.on(eventName, callback);
    }
    once(eventName, callback) {
        this.ioSocket.once(eventName, callback);
    }
    connect() {
        return this.ioSocket.connect();
    }
    disconnect(close) {
        return this.ioSocket.disconnect.apply(this.ioSocket, arguments);
    }
    emit(eventName, ...args) {
        return this.ioSocket.emit.apply(this.ioSocket, arguments);
    }
    removeListener(eventName, callback) {
        return this.ioSocket.removeListener.apply(this.ioSocket, arguments);
    }
    removeAllListeners(eventName) {
        return this.ioSocket.removeAllListeners.apply(this.ioSocket, arguments);
    }
    fromEvent(eventName) {
        if (!this.subscribersCounter[eventName]) {
            this.subscribersCounter[eventName] = 0;
        }
        this.subscribersCounter[eventName]++;
        if (!this.eventObservables$[eventName]) {
            this.eventObservables$[eventName] = new Observable((observer) => {
                const listener = (data) => {
                    observer.next(data);
                };
                this.ioSocket.on(eventName, listener);
                return () => {
                    this.subscribersCounter[eventName]--;
                    if (this.subscribersCounter[eventName] === 0) {
                        this.ioSocket.removeListener(eventName, listener);
                        delete this.eventObservables$[eventName];
                    }
                };
            }).pipe(share());
        }
        return this.eventObservables$[eventName];
    }
    fromOneTimeEvent(eventName) {
        return new Promise(resolve => this.once(eventName, resolve));
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic29ja2V0LWlvLnNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvc29ja2V0LWlvLnNlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQ0EsT0FBTyxFQUFFLFVBQVUsRUFBRSxNQUFNLE1BQU0sQ0FBQztBQUNsQyxPQUFPLEVBQUUsS0FBSyxFQUFFLE1BQU0sZ0JBQWdCLENBQUM7QUFFdkMsT0FBTyxLQUFLLEVBQUUsTUFBTSxrQkFBa0IsQ0FBQztBQUl2QyxNQUFNLE9BQU8sYUFBYTtJQVN0QixZQUFvQixNQUFzQjtRQUF0QixXQUFNLEdBQU4sTUFBTSxDQUFnQjtRQVIxQyx1QkFBa0IsR0FBMkIsRUFBRSxDQUFDO1FBQ2hELHNCQUFpQixHQUFvQyxFQUFFLENBQUM7UUFFeEQsZ0JBQVcsR0FBbUI7WUFDMUIsR0FBRyxFQUFFLEVBQUU7WUFDUCxPQUFPLEVBQUUsRUFBRTtTQUNkLENBQUM7UUFHRSxJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDdEIsTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUM7U0FDN0I7UUFDRCxNQUFNLEdBQUcsR0FBVyxNQUFNLENBQUMsR0FBRyxDQUFDO1FBQy9CLE1BQU0sT0FBTyxHQUFRLE1BQU0sQ0FBQyxPQUFPLENBQUM7UUFDcEMsTUFBTSxNQUFNLEdBQUksRUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUUsRUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQzlELElBQUksQ0FBQyxRQUFRLEdBQUcsTUFBTSxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQsRUFBRSxDQUFDLFNBQWlCO1FBQ2hCLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRCxFQUFFLENBQUMsU0FBaUIsRUFBRSxRQUFrQjtRQUNwQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVELElBQUksQ0FBQyxTQUFpQixFQUFFLFFBQWtCO1FBQ3RDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQsT0FBTztRQUNILE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNuQyxDQUFDO0lBRUQsVUFBVSxDQUFDLEtBQVc7UUFDbEIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQsSUFBSSxDQUFDLFNBQWlCLEVBQUUsR0FBRyxJQUFXO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDOUQsQ0FBQztJQUVELGNBQWMsQ0FBQyxTQUFpQixFQUFFLFFBQW1CO1FBQ2pELE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVELGtCQUFrQixDQUFDLFNBQWtCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQsU0FBUyxDQUFJLFNBQWlCO1FBQzFCLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDckMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUMxQztRQUNELElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDO1FBRXJDLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDcEMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQyxHQUFHLElBQUksVUFBVSxDQUFDLENBQUMsUUFBYSxFQUFFLEVBQUU7Z0JBQ2pFLE1BQU0sUUFBUSxHQUFHLENBQUMsSUFBTyxFQUFFLEVBQUU7b0JBQ3pCLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3hCLENBQUMsQ0FBQztnQkFDRCxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7Z0JBQ3RDLE9BQU8sR0FBRyxFQUFFO29CQUNSLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDO29CQUNyQyxJQUFJLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUU7d0JBQzNDLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQzt3QkFDbEQsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUM7cUJBQzNDO2dCQUNOLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FDSCxLQUFLLEVBQUUsQ0FDVixDQUFDO1NBQ0w7UUFDRCxPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQsZ0JBQWdCLENBQUksU0FBaUI7UUFDakMsT0FBTyxJQUFJLE9BQU8sQ0FBSSxPQUFPLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDcEUsQ0FBQztDQUVKIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSW5qZWN0IH0gZnJvbSAnQGFuZ3VsYXIvY29yZSc7XG5pbXBvcnQgeyBPYnNlcnZhYmxlIH0gZnJvbSAncnhqcyc7XG5pbXBvcnQgeyBzaGFyZSB9IGZyb20gJ3J4anMvb3BlcmF0b3JzJztcblxuaW1wb3J0ICogYXMgaW8gZnJvbSAnc29ja2V0LmlvLWNsaWVudCc7XG5cbmltcG9ydCB7IFNvY2tldElvQ29uZmlnIH0gZnJvbSAnLi9jb25maWcvc29ja2V0LWlvLmNvbmZpZyc7XG5cbmV4cG9ydCBjbGFzcyBXcmFwcGVkU29ja2V0IHtcbiAgICBzdWJzY3JpYmVyc0NvdW50ZXI6IFJlY29yZDxzdHJpbmcsIG51bWJlcj4gPSB7fTtcbiAgICBldmVudE9ic2VydmFibGVzJDogUmVjb3JkPHN0cmluZywgT2JzZXJ2YWJsZTxhbnk+PiA9IHt9O1xuICAgIGlvU29ja2V0OiBhbnk7XG4gICAgZW1wdHlDb25maWc6IFNvY2tldElvQ29uZmlnID0ge1xuICAgICAgICB1cmw6ICcnLFxuICAgICAgICBvcHRpb25zOiB7fVxuICAgIH07XG5cbiAgICBjb25zdHJ1Y3Rvcihwcml2YXRlIGNvbmZpZzogU29ja2V0SW9Db25maWcpIHtcbiAgICAgICAgaWYgKGNvbmZpZyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICBjb25maWcgPSB0aGlzLmVtcHR5Q29uZmlnO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHVybDogc3RyaW5nID0gY29uZmlnLnVybDtcbiAgICAgICAgY29uc3Qgb3B0aW9uczogYW55ID0gY29uZmlnLm9wdGlvbnM7XG4gICAgICAgIGNvbnN0IGlvRnVuYyA9IChpbyBhcyBhbnkpLmRlZmF1bHQgPyAoaW8gYXMgYW55KS5kZWZhdWx0IDogaW87XG4gICAgICAgIHRoaXMuaW9Tb2NrZXQgPSBpb0Z1bmModXJsLCBvcHRpb25zKTtcbiAgICB9XG5cbiAgICBvZihuYW1lc3BhY2U6IHN0cmluZykge1xuICAgICAgICB0aGlzLmlvU29ja2V0Lm9mKG5hbWVzcGFjZSk7XG4gICAgfVxuXG4gICAgb24oZXZlbnROYW1lOiBzdHJpbmcsIGNhbGxiYWNrOiBGdW5jdGlvbikge1xuICAgICAgICB0aGlzLmlvU29ja2V0Lm9uKGV2ZW50TmFtZSwgY2FsbGJhY2spO1xuICAgIH1cblxuICAgIG9uY2UoZXZlbnROYW1lOiBzdHJpbmcsIGNhbGxiYWNrOiBGdW5jdGlvbikge1xuICAgICAgICB0aGlzLmlvU29ja2V0Lm9uY2UoZXZlbnROYW1lLCBjYWxsYmFjayk7XG4gICAgfVxuXG4gICAgY29ubmVjdCgpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuaW9Tb2NrZXQuY29ubmVjdCgpO1xuICAgIH1cblxuICAgIGRpc2Nvbm5lY3QoY2xvc2U/OiBhbnkpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuaW9Tb2NrZXQuZGlzY29ubmVjdC5hcHBseSh0aGlzLmlvU29ja2V0LCBhcmd1bWVudHMpO1xuICAgIH1cblxuICAgIGVtaXQoZXZlbnROYW1lOiBzdHJpbmcsIC4uLmFyZ3M6IGFueVtdKSB7XG4gICAgICAgIHJldHVybiB0aGlzLmlvU29ja2V0LmVtaXQuYXBwbHkodGhpcy5pb1NvY2tldCwgYXJndW1lbnRzKTtcbiAgICB9XG5cbiAgICByZW1vdmVMaXN0ZW5lcihldmVudE5hbWU6IHN0cmluZywgY2FsbGJhY2s/OiBGdW5jdGlvbikge1xuICAgICAgICByZXR1cm4gdGhpcy5pb1NvY2tldC5yZW1vdmVMaXN0ZW5lci5hcHBseSh0aGlzLmlvU29ja2V0LCBhcmd1bWVudHMpO1xuICAgIH1cblxuICAgIHJlbW92ZUFsbExpc3RlbmVycyhldmVudE5hbWU/OiBzdHJpbmcpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuaW9Tb2NrZXQucmVtb3ZlQWxsTGlzdGVuZXJzLmFwcGx5KHRoaXMuaW9Tb2NrZXQsIGFyZ3VtZW50cyk7XG4gICAgfVxuXG4gICAgZnJvbUV2ZW50PFQ+KGV2ZW50TmFtZTogc3RyaW5nKTogT2JzZXJ2YWJsZTxUPiB7XG4gICAgICAgIGlmICghdGhpcy5zdWJzY3JpYmVyc0NvdW50ZXJbZXZlbnROYW1lXSkge1xuICAgICAgICAgICAgdGhpcy5zdWJzY3JpYmVyc0NvdW50ZXJbZXZlbnROYW1lXSA9IDA7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5zdWJzY3JpYmVyc0NvdW50ZXJbZXZlbnROYW1lXSsrO1xuXG4gICAgICAgIGlmICghdGhpcy5ldmVudE9ic2VydmFibGVzJFtldmVudE5hbWVdKSB7XG4gICAgICAgICAgICB0aGlzLmV2ZW50T2JzZXJ2YWJsZXMkW2V2ZW50TmFtZV0gPSBuZXcgT2JzZXJ2YWJsZSgob2JzZXJ2ZXI6IGFueSkgPT4ge1xuICAgICAgICAgICAgICAgIGNvbnN0IGxpc3RlbmVyID0gKGRhdGE6IFQpID0+IHtcbiAgICAgICAgICAgICAgICAgICAgb2JzZXJ2ZXIubmV4dChkYXRhKTtcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICAgICB0aGlzLmlvU29ja2V0Lm9uKGV2ZW50TmFtZSwgbGlzdGVuZXIpO1xuICAgICAgICAgICAgICAgICByZXR1cm4gKCkgPT4ge1xuICAgICAgICAgICAgICAgICAgICAgdGhpcy5zdWJzY3JpYmVyc0NvdW50ZXJbZXZlbnROYW1lXS0tO1xuICAgICAgICAgICAgICAgICAgICAgaWYgKHRoaXMuc3Vic2NyaWJlcnNDb3VudGVyW2V2ZW50TmFtZV0gPT09IDApIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHRoaXMuaW9Tb2NrZXQucmVtb3ZlTGlzdGVuZXIoZXZlbnROYW1lLCBsaXN0ZW5lcik7XG4gICAgICAgICAgICAgICAgICAgICAgICBkZWxldGUgdGhpcy5ldmVudE9ic2VydmFibGVzJFtldmVudE5hbWVdO1xuICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICB9KS5waXBlKFxuICAgICAgICAgICAgICAgIHNoYXJlKClcbiAgICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuZXZlbnRPYnNlcnZhYmxlcyRbZXZlbnROYW1lXTtcbiAgICB9XG5cbiAgICBmcm9tT25lVGltZUV2ZW50PFQ+KGV2ZW50TmFtZTogc3RyaW5nKTogUHJvbWlzZTxUPiB7XG4gICAgICAgIHJldHVybiBuZXcgUHJvbWlzZTxUPihyZXNvbHZlID0+IHRoaXMub25jZShldmVudE5hbWUsIHJlc29sdmUpKTtcbiAgICB9XG5cbn1cbiJdfQ==