import { NgModuleFactory, Compiler, NgModuleFactoryLoader } from '@angular/core';
import { UIRouter, Resolvable, NATIVE_INJECTOR_TOKEN, isString, unnestR, inArray, uniqR, } from '@uirouter/core';
import { UIROUTER_MODULE_TOKEN, UIROUTER_ROOT_MODULE } from '../injectionTokens';
import { applyModuleConfig } from '../uiRouterConfig';
/**
 * Returns a function which lazy loads a nested module
 *
 * This is primarily used by the [[ng2LazyLoadBuilder]] when processing [[Ng2StateDeclaration.loadChildren]].
 *
 * It could also be used manually as a [[StateDeclaration.lazyLoad]] property to lazy load an `NgModule` and its state(s).
 *
 * #### Example:
 * Using `import()` and named export of `HomeModule`
 * ```js
 * declare var System;
 * var futureState = {
 *   name: 'home.**',
 *   url: '/home',
 *   lazyLoad: loadNgModule(() => import('./home/home.module').then(result => result.HomeModule))
 * }
 * ```
 *
 * #### Example:
 * Using a path (string) to the module
 * ```js
 * var futureState = {
 *   name: 'home.**',
 *   url: '/home',
 *   lazyLoad: loadNgModule('./home/home.module#HomeModule')
 * }
 * ```
 *
 *
 * @param moduleToLoad a path (string) to the NgModule to load.
 *    Or a function which loads the NgModule code which should
 *    return a reference to  the `NgModule` class being loaded (or a `Promise` for it).
 *
 * @returns A function which takes a transition, which:
 * - Gets the Injector (scoped properly for the destination state)
 * - Loads and creates the NgModule
 * - Finds the "replacement state" for the target state, and adds the new NgModule Injector to it (as a resolve)
 * - Returns the new states array
 */
export function loadNgModule(moduleToLoad) {
    return function (transition, stateObject) {
        var ng2Injector = transition.injector().get(NATIVE_INJECTOR_TOKEN);
        var createModule = function (factory) { return factory.create(ng2Injector); };
        var applyModule = function (moduleRef) { return applyNgModule(transition, moduleRef, ng2Injector, stateObject); };
        return loadModuleFactory(moduleToLoad, ng2Injector).then(createModule).then(applyModule);
    };
}
/**
 * Returns the module factory that can be used to instantiate a module
 *
 * For strings this:
 * - Finds the correct NgModuleFactoryLoader
 * - Loads the new NgModuleFactory from the path string (async)
 *
 * For a Type<any> or Promise<Type<any>> this:
 * - Compiles the component type (if not running with AOT)
 * - Returns the NgModuleFactory resulting from compilation (or direct loading if using AOT) as a Promise
 *
 * @internal
 */
export function loadModuleFactory(moduleToLoad, ng2Injector) {
    if (isString(moduleToLoad)) {
        return ng2Injector.get(NgModuleFactoryLoader).load(moduleToLoad);
    }
    var compiler = ng2Injector.get(Compiler);
    var unwrapEsModuleDefault = function (x) { return (x && x.__esModule && x['default'] ? x['default'] : x); };
    return Promise.resolve(moduleToLoad())
        .then(unwrapEsModuleDefault)
        .then(function (t) {
        if (t instanceof NgModuleFactory) {
            return t;
        }
        return compiler.compileModuleAsync(t);
    });
}
/**
 * Apply the UI-Router Modules found in the lazy loaded module.
 *
 * Apply the Lazy Loaded NgModule's newly created Injector to the right state in the state tree.
 *
 * Lazy loading uses a placeholder state which is removed (and replaced) after the module is loaded.
 * The NgModule should include a state with the same name as the placeholder.
 *
 * Find the *newly loaded state* with the same name as the *placeholder state*.
 * The NgModule's Injector (and ComponentFactoryResolver) will be added to that state.
 * The Injector/Factory are used when creating Components for the `replacement` state and all its children.
 *
 * @internal
 */
export function applyNgModule(transition, ng2Module, parentInjector, lazyLoadState) {
    var injector = ng2Module.injector;
    var uiRouter = injector.get(UIRouter);
    var registry = uiRouter.stateRegistry;
    var originalName = lazyLoadState.name;
    var originalState = registry.get(originalName);
    // Check if it's a future state (ends with .**)
    var isFuture = /^(.*)\.\*\*$/.exec(originalName);
    // Final name (without the .**)
    var replacementName = isFuture && isFuture[1];
    var newRootModules = multiProviderParentChildDelta(parentInjector, injector, UIROUTER_ROOT_MODULE).reduce(uniqR, []);
    var newChildModules = multiProviderParentChildDelta(parentInjector, injector, UIROUTER_MODULE_TOKEN).reduce(uniqR, []);
    if (newRootModules.length) {
        console.log(newRootModules); // tslint:disable-line:no-console
        throw new Error('Lazy loaded modules should not contain a UIRouterModule.forRoot() module');
    }
    var newStateObjects = newChildModules
        .map(function (module) { return applyModuleConfig(uiRouter, injector, module); })
        .reduce(unnestR, [])
        .reduce(uniqR, []);
    if (isFuture) {
        var replacementState = registry.get(replacementName);
        if (!replacementState || replacementState === originalState) {
            throw new Error("The Future State named '" + originalName + "' lazy loaded an NgModule. " +
                ("The lazy loaded NgModule must have a state named '" + replacementName + "' ") +
                ("which replaces the (placeholder) '" + originalName + "' Future State. ") +
                ("Add a '" + replacementName + "' state to the lazy loaded NgModule ") +
                "using UIRouterModule.forChild({ states: CHILD_STATES }).");
        }
    }
    // Supply the newly loaded states with the Injector from the lazy loaded NgModule.
    // If a tree of states is lazy loaded, only add the injector to the root of the lazy loaded tree.
    // The children will get the injector by resolve inheritance.
    var newParentStates = newStateObjects.filter(function (state) { return !inArray(newStateObjects, state.parent); });
    // Add the Injector to the top of the lazy loaded state tree as a resolve
    newParentStates.forEach(function (state) { return state.resolvables.push(Resolvable.fromData(NATIVE_INJECTOR_TOKEN, injector)); });
    return {};
}
/**
 * Returns the new dependency injection values from the Child Injector
 *
 * When a DI token is defined as multi: true, the child injector
 * can add new values for the token.
 *
 * This function returns the values added by the child injector,  and excludes all values from the parent injector.
 *
 * @internal
 */
export function multiProviderParentChildDelta(parent, child, token) {
    var childVals = child.get(token, []);
    var parentVals = parent.get(token, []);
    return childVals.filter(function (val) { return parentVals.indexOf(val) === -1; });
}
//# sourceMappingURL=data:application/json;base64,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