import { __decorate, __param } from "tslib";
import { Directive, Output, EventEmitter, ContentChildren, Host, Self, Optional } from '@angular/core';
import { UISref } from './uiSref';
import { PathNode, Transition, TargetState, StateObject, anyTrueR, tail, unnestR, Predicate, UIRouterGlobals, Param, PathUtils, identity, uniqR, } from '@uirouter/core';
import { BehaviorSubject, of, from, combineLatest, concat } from 'rxjs';
import { switchMap, map } from 'rxjs/operators';
/** @internal */
var inactiveStatus = {
    active: false,
    exact: false,
    entering: false,
    exiting: false,
    targetStates: [],
};
/**
 * Returns a Predicate<PathNode[]>
 *
 * The predicate returns true when the target state (and param values)
 * match the (tail of) the path, and the path's param values
 *
 * @internal
 */
var pathMatches = function (target) {
    if (!target.exists())
        return function () { return false; };
    var state = target.$state();
    var targetParamVals = target.params();
    var targetPath = PathUtils.buildPath(target);
    var paramSchema = targetPath
        .map(function (node) { return node.paramSchema; })
        .reduce(unnestR, [])
        .filter(function (param) { return targetParamVals.hasOwnProperty(param.id); });
    return function (path) {
        var tailNode = tail(path);
        if (!tailNode || tailNode.state !== state)
            return false;
        var paramValues = PathUtils.paramValues(path);
        return Param.equals(paramSchema, paramValues, targetParamVals);
    };
};
var ɵ0 = pathMatches;
/**
 * Given basePath: [a, b], appendPath: [c, d]),
 * Expands the path to [c], [c, d]
 * Then appends each to [a,b,] and returns: [a, b, c], [a, b, c, d]
 *
 * @internal
 */
function spreadToSubPaths(basePath, appendPath) {
    return appendPath.map(function (node) { return basePath.concat(PathUtils.subPath(appendPath, function (n) { return n.state === node.state; })); });
}
/**
 * Given a TransEvt (Transition event: started, success, error)
 * and a UISref Target State, return a SrefStatus object
 * which represents the current status of that Sref:
 * active, activeEq (exact match), entering, exiting
 *
 * @internal
 */
function getSrefStatus(event, srefTarget) {
    var pathMatchesTarget = pathMatches(srefTarget);
    var tc = event.trans.treeChanges();
    var isStartEvent = event.evt === 'start';
    var isSuccessEvent = event.evt === 'success';
    var activePath = isSuccessEvent ? tc.to : tc.from;
    var isActive = function () { return spreadToSubPaths([], activePath).map(pathMatchesTarget).reduce(anyTrueR, false); };
    var isExact = function () { return pathMatchesTarget(activePath); };
    var isEntering = function () { return spreadToSubPaths(tc.retained, tc.entering).map(pathMatchesTarget).reduce(anyTrueR, false); };
    var isExiting = function () { return spreadToSubPaths(tc.retained, tc.exiting).map(pathMatchesTarget).reduce(anyTrueR, false); };
    return {
        active: isActive(),
        exact: isExact(),
        entering: isStartEvent ? isEntering() : false,
        exiting: isStartEvent ? isExiting() : false,
        targetStates: [srefTarget],
    };
}
/** @internal */
function mergeSrefStatus(left, right) {
    return {
        active: left.active || right.active,
        exact: left.exact || right.exact,
        entering: left.entering || right.entering,
        exiting: left.exiting || right.exiting,
        targetStates: left.targetStates.concat(right.targetStates),
    };
}
/**
 * A directive which emits events when a paired [[UISref]] status changes.
 *
 * This directive is primarily used by the [[UISrefActive]] directives to monitor `UISref`(s).
 *
 * This directive shares two attribute selectors with `UISrefActive`:
 *
 * - `[uiSrefActive]`
 * - `[uiSrefActiveEq]`.
 *
 * Thus, whenever a `UISrefActive` directive is created, a `UISrefStatus` directive is also created.
 *
 * Most apps should simply use `UISrefActive`, but some advanced components may want to process the
 * [[SrefStatus]] events directly.
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatusChanged($event)">
 *   <a uiSref="book" [uiParams]="{ bookId: book.id }">Book {{ book.name }}</a>
 * </li>
 * ```
 *
 * The `uiSrefStatus` event is emitted whenever an enclosed `uiSref`'s status changes.
 * The event emitted is of type [[SrefStatus]], and has boolean values for `active`, `exact`, `entering`, and `exiting`; also has a [[StateOrName]] `identifier`value.
 *
 * The values from this event can be captured and stored on a component (then applied, e.g., using ngClass).
 *
 * ---
 *
 * A single `uiSrefStatus` can enclose multiple `uiSref`.
 * Each status boolean (`active`, `exact`, `entering`, `exiting`) will be true if *any of the enclosed `uiSref` status is true*.
 * In other words, all enclosed `uiSref` statuses  are merged to a single status using `||` (logical or).
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatus($event)" uiSref="admin">
 *   Home
 *   <ul>
 *     <li> <a uiSref="admin.users">Users</a> </li>
 *     <li> <a uiSref="admin.groups">Groups</a> </li>
 *   </ul>
 * </li>
 * ```
 *
 * In the above example, `$event.active === true` when either `admin.users` or `admin.groups` is active.
 *
 * ---
 *
 * This API is subject to change.
 */
var UISrefStatus = /** @class */ (function () {
    function UISrefStatus(_hostUiSref, _globals) {
        /** current statuses of the state/params the uiSref directive is linking to */
        this.uiSrefStatus = new EventEmitter(false);
        this._globals = _globals;
        this._hostUiSref = _hostUiSref;
        this.status = Object.assign({}, inactiveStatus);
    }
    UISrefStatus.prototype.ngAfterContentInit = function () {
        var _this = this;
        // Map each transition start event to a stream of:
        // start -> (success|error)
        var transEvents$ = this._globals.start$.pipe(switchMap(function (trans) {
            var event = function (evt) { return ({ evt: evt, trans: trans }); };
            var transStart$ = of(event('start'));
            var transResult = trans.promise.then(function () { return event('success'); }, function () { return event('error'); });
            var transFinish$ = from(transResult);
            return concat(transStart$, transFinish$);
        }));
        var withHostSref = function (childrenSrefs) {
            return childrenSrefs.concat(_this._hostUiSref).filter(identity).reduce(uniqR, []);
        };
        // Watch the @ContentChildren UISref[] components and get their target states
        this._srefs$ = new BehaviorSubject(withHostSref(this._srefs.toArray()));
        this._srefChangesSub = this._srefs.changes.subscribe(function (srefs) {
            return _this._srefs$.next(withHostSref(srefs.toArray()));
        });
        var targetStates$ = this._srefs$.pipe(switchMap(function (srefs) { return combineLatest(srefs.map(function (sref) { return sref.targetState$; })); }));
        // Calculate the status of each UISref based on the transition event.
        // Reduce the statuses (if multiple) by or-ing each flag.
        this._subscription = transEvents$
            .pipe(switchMap(function (evt) {
            return targetStates$.pipe(map(function (targets) {
                var statuses = targets.map(function (target) { return getSrefStatus(evt, target); });
                return statuses.reduce(mergeSrefStatus);
            }));
        }))
            .subscribe(this._setStatus.bind(this));
    };
    UISrefStatus.prototype.ngOnDestroy = function () {
        if (this._subscription)
            this._subscription.unsubscribe();
        if (this._srefChangesSub)
            this._srefChangesSub.unsubscribe();
        if (this._srefs$)
            this._srefs$.unsubscribe();
        this._subscription = this._srefChangesSub = this._srefs$ = undefined;
    };
    UISrefStatus.prototype._setStatus = function (status) {
        this.status = status;
        this.uiSrefStatus.emit(status);
    };
    UISrefStatus.ctorParameters = function () { return [
        { type: UISref, decorators: [{ type: Host }, { type: Self }, { type: Optional }] },
        { type: UIRouterGlobals }
    ]; };
    __decorate([
        Output('uiSrefStatus')
    ], UISrefStatus.prototype, "uiSrefStatus", void 0);
    __decorate([
        ContentChildren(UISref, { descendants: true })
    ], UISrefStatus.prototype, "_srefs", void 0);
    UISrefStatus = __decorate([
        Directive({
            selector: '[uiSrefStatus],[uiSrefActive],[uiSrefActiveEq]',
            exportAs: 'uiSrefStatus',
        }),
        __param(0, Host()), __param(0, Self()), __param(0, Optional())
    ], UISrefStatus);
    return UISrefStatus;
}());
export { UISrefStatus };
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidWlTcmVmU3RhdHVzLmpzIiwic291cmNlUm9vdCI6Im5nOi8vQHVpcm91dGVyL2FuZ3VsYXIvIiwic291cmNlcyI6WyJkaXJlY3RpdmVzL3VpU3JlZlN0YXR1cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsT0FBTyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFLGVBQWUsRUFBYSxJQUFJLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxNQUFNLGVBQWUsQ0FBQztBQUNsSCxPQUFPLEVBQUUsTUFBTSxFQUFFLE1BQU0sVUFBVSxDQUFDO0FBQ2xDLE9BQU8sRUFDTCxRQUFRLEVBQ1IsVUFBVSxFQUNWLFdBQVcsRUFDWCxXQUFXLEVBQ1gsUUFBUSxFQUNSLElBQUksRUFDSixPQUFPLEVBQ1AsU0FBUyxFQUNULGVBQWUsRUFDZixLQUFLLEVBQ0wsU0FBUyxFQUNULFFBQVEsRUFDUixLQUFLLEdBQ04sTUFBTSxnQkFBZ0IsQ0FBQztBQUV4QixPQUFPLEVBQTRCLGVBQWUsRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLGFBQWEsRUFBRSxNQUFNLEVBQUUsTUFBTSxNQUFNLENBQUM7QUFDbEcsT0FBTyxFQUFFLFNBQVMsRUFBRSxHQUFHLEVBQU8sTUFBTSxnQkFBZ0IsQ0FBQztBQXdCckQsZ0JBQWdCO0FBQ2hCLElBQU0sY0FBYyxHQUFlO0lBQ2pDLE1BQU0sRUFBRSxLQUFLO0lBQ2IsS0FBSyxFQUFFLEtBQUs7SUFDWixRQUFRLEVBQUUsS0FBSztJQUNmLE9BQU8sRUFBRSxLQUFLO0lBQ2QsWUFBWSxFQUFFLEVBQUU7Q0FDakIsQ0FBQztBQUVGOzs7Ozs7O0dBT0c7QUFDSCxJQUFNLFdBQVcsR0FBRyxVQUFDLE1BQW1CO0lBQ3RDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFO1FBQUUsT0FBTyxjQUFNLE9BQUEsS0FBSyxFQUFMLENBQUssQ0FBQztJQUN6QyxJQUFNLEtBQUssR0FBZ0IsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBQzNDLElBQU0sZUFBZSxHQUFHLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQztJQUN4QyxJQUFNLFVBQVUsR0FBZSxTQUFTLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzNELElBQU0sV0FBVyxHQUFZLFVBQVU7U0FDcEMsR0FBRyxDQUFDLFVBQUMsSUFBSSxJQUFLLE9BQUEsSUFBSSxDQUFDLFdBQVcsRUFBaEIsQ0FBZ0IsQ0FBQztTQUMvQixNQUFNLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQztTQUNuQixNQUFNLENBQUMsVUFBQyxLQUFZLElBQUssT0FBQSxlQUFlLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsRUFBeEMsQ0FBd0MsQ0FBQyxDQUFDO0lBRXRFLE9BQU8sVUFBQyxJQUFnQjtRQUN0QixJQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDNUIsSUFBSSxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsS0FBSyxLQUFLLEtBQUs7WUFBRSxPQUFPLEtBQUssQ0FBQztRQUN4RCxJQUFNLFdBQVcsR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2hELE9BQU8sS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsV0FBVyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBQ2pFLENBQUMsQ0FBQztBQUNKLENBQUMsQ0FBQzs7QUFFRjs7Ozs7O0dBTUc7QUFDSCxTQUFTLGdCQUFnQixDQUFDLFFBQW9CLEVBQUUsVUFBc0I7SUFDcEUsT0FBTyxVQUFVLENBQUMsR0FBRyxDQUFDLFVBQUMsSUFBSSxJQUFLLE9BQUEsUUFBUSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRSxVQUFDLENBQUMsSUFBSyxPQUFBLENBQUMsQ0FBQyxLQUFLLEtBQUssSUFBSSxDQUFDLEtBQUssRUFBdEIsQ0FBc0IsQ0FBQyxDQUFDLEVBQTdFLENBQTZFLENBQUMsQ0FBQztBQUNqSCxDQUFDO0FBRUQ7Ozs7Ozs7R0FPRztBQUNILFNBQVMsYUFBYSxDQUFDLEtBQWUsRUFBRSxVQUF1QjtJQUM3RCxJQUFNLGlCQUFpQixHQUFHLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNsRCxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBRXJDLElBQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxHQUFHLEtBQUssT0FBTyxDQUFDO0lBQzNDLElBQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxHQUFHLEtBQUssU0FBUyxDQUFDO0lBQy9DLElBQU0sVUFBVSxHQUFlLGNBQWMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQztJQUVoRSxJQUFNLFFBQVEsR0FBRyxjQUFNLE9BQUEsZ0JBQWdCLENBQUMsRUFBRSxFQUFFLFVBQVUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLEVBQS9FLENBQStFLENBQUM7SUFFdkcsSUFBTSxPQUFPLEdBQUcsY0FBTSxPQUFBLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxFQUE3QixDQUE2QixDQUFDO0lBRXBELElBQU0sVUFBVSxHQUFHLGNBQU0sT0FBQSxnQkFBZ0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxFQUF6RixDQUF5RixDQUFDO0lBRW5ILElBQU0sU0FBUyxHQUFHLGNBQU0sT0FBQSxnQkFBZ0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxFQUF4RixDQUF3RixDQUFDO0lBRWpILE9BQU87UUFDTCxNQUFNLEVBQUUsUUFBUSxFQUFFO1FBQ2xCLEtBQUssRUFBRSxPQUFPLEVBQUU7UUFDaEIsUUFBUSxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUs7UUFDN0MsT0FBTyxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUs7UUFDM0MsWUFBWSxFQUFFLENBQUMsVUFBVSxDQUFDO0tBQ2IsQ0FBQztBQUNsQixDQUFDO0FBRUQsZ0JBQWdCO0FBQ2hCLFNBQVMsZUFBZSxDQUFDLElBQWdCLEVBQUUsS0FBaUI7SUFDMUQsT0FBTztRQUNMLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxNQUFNO1FBQ25DLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxLQUFLO1FBQ2hDLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxJQUFJLEtBQUssQ0FBQyxRQUFRO1FBQ3pDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxJQUFJLEtBQUssQ0FBQyxPQUFPO1FBQ3RDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO0tBQzNELENBQUM7QUFDSixDQUFDO0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBK0NHO0FBS0g7SUFlRSxzQkFBd0MsV0FBbUIsRUFBRSxRQUF5QjtRQWR0Riw4RUFBOEU7UUFDdEQsaUJBQVksR0FBRyxJQUFJLFlBQVksQ0FBYSxLQUFLLENBQUMsQ0FBQztRQWN6RSxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUN6QixJQUFJLENBQUMsV0FBVyxHQUFHLFdBQVcsQ0FBQztRQUMvQixJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLGNBQWMsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRCx5Q0FBa0IsR0FBbEI7UUFBQSxpQkE2Q0M7UUE1Q0Msa0RBQWtEO1FBQ2xELDJCQUEyQjtRQUMzQixJQUFNLFlBQVksR0FBeUIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNsRSxTQUFTLENBQUMsVUFBQyxLQUFpQjtZQUMxQixJQUFNLEtBQUssR0FBRyxVQUFDLEdBQVcsSUFBSyxPQUFBLENBQUMsRUFBRSxHQUFHLEtBQUEsRUFBRSxLQUFLLE9BQUEsRUFBZSxDQUFBLEVBQTVCLENBQTRCLENBQUM7WUFFNUQsSUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ3ZDLElBQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUNwQyxjQUFNLE9BQUEsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFoQixDQUFnQixFQUN0QixjQUFNLE9BQUEsS0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFkLENBQWMsQ0FDckIsQ0FBQztZQUNGLElBQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUV2QyxPQUFPLE1BQU0sQ0FBQyxXQUFXLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDM0MsQ0FBQyxDQUFDLENBQ0gsQ0FBQztRQUVGLElBQU0sWUFBWSxHQUFHLFVBQUMsYUFBdUI7WUFDM0MsT0FBQSxhQUFhLENBQUMsTUFBTSxDQUFDLEtBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUM7UUFBekUsQ0FBeUUsQ0FBQztRQUU1RSw2RUFBNkU7UUFDN0UsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLGVBQWUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDeEUsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsVUFBQyxLQUF3QjtZQUM1RSxPQUFBLEtBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUFoRCxDQUFnRCxDQUNqRCxDQUFDO1FBRUYsSUFBTSxhQUFhLEdBQThCLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUNoRSxTQUFTLENBQUMsVUFBQyxLQUFlLElBQUssT0FBQSxhQUFhLENBQWdCLEtBQUssQ0FBQyxHQUFHLENBQUMsVUFBQyxJQUFJLElBQUssT0FBQSxJQUFJLENBQUMsWUFBWSxFQUFqQixDQUFpQixDQUFDLENBQUMsRUFBcEUsQ0FBb0UsQ0FBQyxDQUNyRyxDQUFDO1FBRUYscUVBQXFFO1FBQ3JFLHlEQUF5RDtRQUN6RCxJQUFJLENBQUMsYUFBYSxHQUFHLFlBQVk7YUFDOUIsSUFBSSxDQUNILFNBQVMsQ0FBQyxVQUFDLEdBQWE7WUFDdEIsT0FBTyxhQUFhLENBQUMsSUFBSSxDQUN2QixHQUFHLENBQUMsVUFBQyxPQUFzQjtnQkFDekIsSUFBTSxRQUFRLEdBQWlCLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBQyxNQUFNLElBQUssT0FBQSxhQUFhLENBQUMsR0FBRyxFQUFFLE1BQU0sQ0FBQyxFQUExQixDQUEwQixDQUFDLENBQUM7Z0JBQ25GLE9BQU8sUUFBUSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUMxQyxDQUFDLENBQUMsQ0FDSCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQ0g7YUFDQSxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQsa0NBQVcsR0FBWDtRQUNFLElBQUksSUFBSSxDQUFDLGFBQWE7WUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQ3pELElBQUksSUFBSSxDQUFDLGVBQWU7WUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQzdELElBQUksSUFBSSxDQUFDLE9BQU87WUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQzdDLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsT0FBTyxHQUFHLFNBQVMsQ0FBQztJQUN2RSxDQUFDO0lBRU8saUNBQVUsR0FBbEIsVUFBbUIsTUFBa0I7UUFDbkMsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUM7UUFDckIsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDakMsQ0FBQzs7Z0JBL0RvRCxNQUFNLHVCQUE5QyxJQUFJLFlBQUksSUFBSSxZQUFJLFFBQVE7Z0JBQWtDLGVBQWU7O0lBYjlEO1FBQXZCLE1BQU0sQ0FBQyxjQUFjLENBQUM7c0RBQW9EO0lBRzNFO1FBREMsZUFBZSxDQUFDLE1BQU0sRUFBRSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsQ0FBQztnREFDYjtJQUx2QixZQUFZO1FBSnhCLFNBQVMsQ0FBQztZQUNULFFBQVEsRUFBRSxnREFBZ0Q7WUFDMUQsUUFBUSxFQUFFLGNBQWM7U0FDekIsQ0FBQztRQWdCYSxXQUFBLElBQUksRUFBRSxDQUFBLEVBQUUsV0FBQSxJQUFJLEVBQUUsQ0FBQSxFQUFFLFdBQUEsUUFBUSxFQUFFLENBQUE7T0FmNUIsWUFBWSxDQStFeEI7SUFBRCxtQkFBQztDQUFBLEFBL0VELElBK0VDO1NBL0VZLFlBQVkiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBEaXJlY3RpdmUsIE91dHB1dCwgRXZlbnRFbWl0dGVyLCBDb250ZW50Q2hpbGRyZW4sIFF1ZXJ5TGlzdCwgSG9zdCwgU2VsZiwgT3B0aW9uYWwgfSBmcm9tICdAYW5ndWxhci9jb3JlJztcbmltcG9ydCB7IFVJU3JlZiB9IGZyb20gJy4vdWlTcmVmJztcbmltcG9ydCB7XG4gIFBhdGhOb2RlLFxuICBUcmFuc2l0aW9uLFxuICBUYXJnZXRTdGF0ZSxcbiAgU3RhdGVPYmplY3QsXG4gIGFueVRydWVSLFxuICB0YWlsLFxuICB1bm5lc3RSLFxuICBQcmVkaWNhdGUsXG4gIFVJUm91dGVyR2xvYmFscyxcbiAgUGFyYW0sXG4gIFBhdGhVdGlscyxcbiAgaWRlbnRpdHksXG4gIHVuaXFSLFxufSBmcm9tICdAdWlyb3V0ZXIvY29yZSc7XG5cbmltcG9ydCB7IFN1YnNjcmlwdGlvbiwgT2JzZXJ2YWJsZSwgQmVoYXZpb3JTdWJqZWN0LCBvZiwgZnJvbSwgY29tYmluZUxhdGVzdCwgY29uY2F0IH0gZnJvbSAncnhqcyc7XG5pbXBvcnQgeyBzd2l0Y2hNYXAsIG1hcCwgdGFwIH0gZnJvbSAncnhqcy9vcGVyYXRvcnMnO1xuXG4vKiogQGludGVybmFsICovXG5pbnRlcmZhY2UgVHJhbnNFdnQge1xuICBldnQ6IHN0cmluZztcbiAgdHJhbnM6IFRyYW5zaXRpb247XG59XG5cbi8qKlxuICogVUlTcmVmIHN0YXR1cyBlbWl0dGVkIGZyb20gW1tVSVNyZWZTdGF0dXNdXVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNyZWZTdGF0dXMge1xuICAvKiogVGhlIHNyZWYncyB0YXJnZXQgc3RhdGUgKG9yIG9uZSBvZiBpdHMgY2hpbGRyZW4pIGlzIGN1cnJlbnRseSBhY3RpdmUgKi9cbiAgYWN0aXZlOiBib29sZWFuO1xuICAvKiogVGhlIHNyZWYncyB0YXJnZXQgc3RhdGUgaXMgY3VycmVudGx5IGFjdGl2ZSAqL1xuICBleGFjdDogYm9vbGVhbjtcbiAgLyoqIEEgdHJhbnNpdGlvbiBpcyBlbnRlcmluZyB0aGUgc3JlZidzIHRhcmdldCBzdGF0ZSAqL1xuICBlbnRlcmluZzogYm9vbGVhbjtcbiAgLyoqIEEgdHJhbnNpdGlvbiBpcyBleGl0aW5nIHRoZSBzcmVmJ3MgdGFyZ2V0IHN0YXRlICovXG4gIGV4aXRpbmc6IGJvb2xlYW47XG4gIC8qKiBUaGUgZW5jbG9zZWQgc3JlZihzKSB0YXJnZXQgc3RhdGUocykgKi9cbiAgdGFyZ2V0U3RhdGVzOiBUYXJnZXRTdGF0ZVtdO1xufVxuXG4vKiogQGludGVybmFsICovXG5jb25zdCBpbmFjdGl2ZVN0YXR1czogU3JlZlN0YXR1cyA9IHtcbiAgYWN0aXZlOiBmYWxzZSxcbiAgZXhhY3Q6IGZhbHNlLFxuICBlbnRlcmluZzogZmFsc2UsXG4gIGV4aXRpbmc6IGZhbHNlLFxuICB0YXJnZXRTdGF0ZXM6IFtdLFxufTtcblxuLyoqXG4gKiBSZXR1cm5zIGEgUHJlZGljYXRlPFBhdGhOb2RlW10+XG4gKlxuICogVGhlIHByZWRpY2F0ZSByZXR1cm5zIHRydWUgd2hlbiB0aGUgdGFyZ2V0IHN0YXRlIChhbmQgcGFyYW0gdmFsdWVzKVxuICogbWF0Y2ggdGhlICh0YWlsIG9mKSB0aGUgcGF0aCwgYW5kIHRoZSBwYXRoJ3MgcGFyYW0gdmFsdWVzXG4gKlxuICogQGludGVybmFsXG4gKi9cbmNvbnN0IHBhdGhNYXRjaGVzID0gKHRhcmdldDogVGFyZ2V0U3RhdGUpOiBQcmVkaWNhdGU8UGF0aE5vZGVbXT4gPT4ge1xuICBpZiAoIXRhcmdldC5leGlzdHMoKSkgcmV0dXJuICgpID0+IGZhbHNlO1xuICBjb25zdCBzdGF0ZTogU3RhdGVPYmplY3QgPSB0YXJnZXQuJHN0YXRlKCk7XG4gIGNvbnN0IHRhcmdldFBhcmFtVmFscyA9IHRhcmdldC5wYXJhbXMoKTtcbiAgY29uc3QgdGFyZ2V0UGF0aDogUGF0aE5vZGVbXSA9IFBhdGhVdGlscy5idWlsZFBhdGgodGFyZ2V0KTtcbiAgY29uc3QgcGFyYW1TY2hlbWE6IFBhcmFtW10gPSB0YXJnZXRQYXRoXG4gICAgLm1hcCgobm9kZSkgPT4gbm9kZS5wYXJhbVNjaGVtYSlcbiAgICAucmVkdWNlKHVubmVzdFIsIFtdKVxuICAgIC5maWx0ZXIoKHBhcmFtOiBQYXJhbSkgPT4gdGFyZ2V0UGFyYW1WYWxzLmhhc093blByb3BlcnR5KHBhcmFtLmlkKSk7XG5cbiAgcmV0dXJuIChwYXRoOiBQYXRoTm9kZVtdKSA9PiB7XG4gICAgY29uc3QgdGFpbE5vZGUgPSB0YWlsKHBhdGgpO1xuICAgIGlmICghdGFpbE5vZGUgfHwgdGFpbE5vZGUuc3RhdGUgIT09IHN0YXRlKSByZXR1cm4gZmFsc2U7XG4gICAgY29uc3QgcGFyYW1WYWx1ZXMgPSBQYXRoVXRpbHMucGFyYW1WYWx1ZXMocGF0aCk7XG4gICAgcmV0dXJuIFBhcmFtLmVxdWFscyhwYXJhbVNjaGVtYSwgcGFyYW1WYWx1ZXMsIHRhcmdldFBhcmFtVmFscyk7XG4gIH07XG59O1xuXG4vKipcbiAqIEdpdmVuIGJhc2VQYXRoOiBbYSwgYl0sIGFwcGVuZFBhdGg6IFtjLCBkXSksXG4gKiBFeHBhbmRzIHRoZSBwYXRoIHRvIFtjXSwgW2MsIGRdXG4gKiBUaGVuIGFwcGVuZHMgZWFjaCB0byBbYSxiLF0gYW5kIHJldHVybnM6IFthLCBiLCBjXSwgW2EsIGIsIGMsIGRdXG4gKlxuICogQGludGVybmFsXG4gKi9cbmZ1bmN0aW9uIHNwcmVhZFRvU3ViUGF0aHMoYmFzZVBhdGg6IFBhdGhOb2RlW10sIGFwcGVuZFBhdGg6IFBhdGhOb2RlW10pOiBQYXRoTm9kZVtdW10ge1xuICByZXR1cm4gYXBwZW5kUGF0aC5tYXAoKG5vZGUpID0+IGJhc2VQYXRoLmNvbmNhdChQYXRoVXRpbHMuc3ViUGF0aChhcHBlbmRQYXRoLCAobikgPT4gbi5zdGF0ZSA9PT0gbm9kZS5zdGF0ZSkpKTtcbn1cblxuLyoqXG4gKiBHaXZlbiBhIFRyYW5zRXZ0IChUcmFuc2l0aW9uIGV2ZW50OiBzdGFydGVkLCBzdWNjZXNzLCBlcnJvcilcbiAqIGFuZCBhIFVJU3JlZiBUYXJnZXQgU3RhdGUsIHJldHVybiBhIFNyZWZTdGF0dXMgb2JqZWN0XG4gKiB3aGljaCByZXByZXNlbnRzIHRoZSBjdXJyZW50IHN0YXR1cyBvZiB0aGF0IFNyZWY6XG4gKiBhY3RpdmUsIGFjdGl2ZUVxIChleGFjdCBtYXRjaCksIGVudGVyaW5nLCBleGl0aW5nXG4gKlxuICogQGludGVybmFsXG4gKi9cbmZ1bmN0aW9uIGdldFNyZWZTdGF0dXMoZXZlbnQ6IFRyYW5zRXZ0LCBzcmVmVGFyZ2V0OiBUYXJnZXRTdGF0ZSk6IFNyZWZTdGF0dXMge1xuICBjb25zdCBwYXRoTWF0Y2hlc1RhcmdldCA9IHBhdGhNYXRjaGVzKHNyZWZUYXJnZXQpO1xuICBjb25zdCB0YyA9IGV2ZW50LnRyYW5zLnRyZWVDaGFuZ2VzKCk7XG5cbiAgY29uc3QgaXNTdGFydEV2ZW50ID0gZXZlbnQuZXZ0ID09PSAnc3RhcnQnO1xuICBjb25zdCBpc1N1Y2Nlc3NFdmVudCA9IGV2ZW50LmV2dCA9PT0gJ3N1Y2Nlc3MnO1xuICBjb25zdCBhY3RpdmVQYXRoOiBQYXRoTm9kZVtdID0gaXNTdWNjZXNzRXZlbnQgPyB0Yy50byA6IHRjLmZyb207XG5cbiAgY29uc3QgaXNBY3RpdmUgPSAoKSA9PiBzcHJlYWRUb1N1YlBhdGhzKFtdLCBhY3RpdmVQYXRoKS5tYXAocGF0aE1hdGNoZXNUYXJnZXQpLnJlZHVjZShhbnlUcnVlUiwgZmFsc2UpO1xuXG4gIGNvbnN0IGlzRXhhY3QgPSAoKSA9PiBwYXRoTWF0Y2hlc1RhcmdldChhY3RpdmVQYXRoKTtcblxuICBjb25zdCBpc0VudGVyaW5nID0gKCkgPT4gc3ByZWFkVG9TdWJQYXRocyh0Yy5yZXRhaW5lZCwgdGMuZW50ZXJpbmcpLm1hcChwYXRoTWF0Y2hlc1RhcmdldCkucmVkdWNlKGFueVRydWVSLCBmYWxzZSk7XG5cbiAgY29uc3QgaXNFeGl0aW5nID0gKCkgPT4gc3ByZWFkVG9TdWJQYXRocyh0Yy5yZXRhaW5lZCwgdGMuZXhpdGluZykubWFwKHBhdGhNYXRjaGVzVGFyZ2V0KS5yZWR1Y2UoYW55VHJ1ZVIsIGZhbHNlKTtcblxuICByZXR1cm4ge1xuICAgIGFjdGl2ZTogaXNBY3RpdmUoKSxcbiAgICBleGFjdDogaXNFeGFjdCgpLFxuICAgIGVudGVyaW5nOiBpc1N0YXJ0RXZlbnQgPyBpc0VudGVyaW5nKCkgOiBmYWxzZSxcbiAgICBleGl0aW5nOiBpc1N0YXJ0RXZlbnQgPyBpc0V4aXRpbmcoKSA6IGZhbHNlLFxuICAgIHRhcmdldFN0YXRlczogW3NyZWZUYXJnZXRdLFxuICB9IGFzIFNyZWZTdGF0dXM7XG59XG5cbi8qKiBAaW50ZXJuYWwgKi9cbmZ1bmN0aW9uIG1lcmdlU3JlZlN0YXR1cyhsZWZ0OiBTcmVmU3RhdHVzLCByaWdodDogU3JlZlN0YXR1cyk6IFNyZWZTdGF0dXMge1xuICByZXR1cm4ge1xuICAgIGFjdGl2ZTogbGVmdC5hY3RpdmUgfHwgcmlnaHQuYWN0aXZlLFxuICAgIGV4YWN0OiBsZWZ0LmV4YWN0IHx8IHJpZ2h0LmV4YWN0LFxuICAgIGVudGVyaW5nOiBsZWZ0LmVudGVyaW5nIHx8IHJpZ2h0LmVudGVyaW5nLFxuICAgIGV4aXRpbmc6IGxlZnQuZXhpdGluZyB8fCByaWdodC5leGl0aW5nLFxuICAgIHRhcmdldFN0YXRlczogbGVmdC50YXJnZXRTdGF0ZXMuY29uY2F0KHJpZ2h0LnRhcmdldFN0YXRlcyksXG4gIH07XG59XG5cbi8qKlxuICogQSBkaXJlY3RpdmUgd2hpY2ggZW1pdHMgZXZlbnRzIHdoZW4gYSBwYWlyZWQgW1tVSVNyZWZdXSBzdGF0dXMgY2hhbmdlcy5cbiAqXG4gKiBUaGlzIGRpcmVjdGl2ZSBpcyBwcmltYXJpbHkgdXNlZCBieSB0aGUgW1tVSVNyZWZBY3RpdmVdXSBkaXJlY3RpdmVzIHRvIG1vbml0b3IgYFVJU3JlZmAocykuXG4gKlxuICogVGhpcyBkaXJlY3RpdmUgc2hhcmVzIHR3byBhdHRyaWJ1dGUgc2VsZWN0b3JzIHdpdGggYFVJU3JlZkFjdGl2ZWA6XG4gKlxuICogLSBgW3VpU3JlZkFjdGl2ZV1gXG4gKiAtIGBbdWlTcmVmQWN0aXZlRXFdYC5cbiAqXG4gKiBUaHVzLCB3aGVuZXZlciBhIGBVSVNyZWZBY3RpdmVgIGRpcmVjdGl2ZSBpcyBjcmVhdGVkLCBhIGBVSVNyZWZTdGF0dXNgIGRpcmVjdGl2ZSBpcyBhbHNvIGNyZWF0ZWQuXG4gKlxuICogTW9zdCBhcHBzIHNob3VsZCBzaW1wbHkgdXNlIGBVSVNyZWZBY3RpdmVgLCBidXQgc29tZSBhZHZhbmNlZCBjb21wb25lbnRzIG1heSB3YW50IHRvIHByb2Nlc3MgdGhlXG4gKiBbW1NyZWZTdGF0dXNdXSBldmVudHMgZGlyZWN0bHkuXG4gKlxuICogYGBganNcbiAqIDxsaSAodWlTcmVmU3RhdHVzKT1cIm9uU3JlZlN0YXR1c0NoYW5nZWQoJGV2ZW50KVwiPlxuICogICA8YSB1aVNyZWY9XCJib29rXCIgW3VpUGFyYW1zXT1cInsgYm9va0lkOiBib29rLmlkIH1cIj5Cb29rIHt7IGJvb2submFtZSB9fTwvYT5cbiAqIDwvbGk+XG4gKiBgYGBcbiAqXG4gKiBUaGUgYHVpU3JlZlN0YXR1c2AgZXZlbnQgaXMgZW1pdHRlZCB3aGVuZXZlciBhbiBlbmNsb3NlZCBgdWlTcmVmYCdzIHN0YXR1cyBjaGFuZ2VzLlxuICogVGhlIGV2ZW50IGVtaXR0ZWQgaXMgb2YgdHlwZSBbW1NyZWZTdGF0dXNdXSwgYW5kIGhhcyBib29sZWFuIHZhbHVlcyBmb3IgYGFjdGl2ZWAsIGBleGFjdGAsIGBlbnRlcmluZ2AsIGFuZCBgZXhpdGluZ2A7IGFsc28gaGFzIGEgW1tTdGF0ZU9yTmFtZV1dIGBpZGVudGlmaWVyYHZhbHVlLlxuICpcbiAqIFRoZSB2YWx1ZXMgZnJvbSB0aGlzIGV2ZW50IGNhbiBiZSBjYXB0dXJlZCBhbmQgc3RvcmVkIG9uIGEgY29tcG9uZW50ICh0aGVuIGFwcGxpZWQsIGUuZy4sIHVzaW5nIG5nQ2xhc3MpLlxuICpcbiAqIC0tLVxuICpcbiAqIEEgc2luZ2xlIGB1aVNyZWZTdGF0dXNgIGNhbiBlbmNsb3NlIG11bHRpcGxlIGB1aVNyZWZgLlxuICogRWFjaCBzdGF0dXMgYm9vbGVhbiAoYGFjdGl2ZWAsIGBleGFjdGAsIGBlbnRlcmluZ2AsIGBleGl0aW5nYCkgd2lsbCBiZSB0cnVlIGlmICphbnkgb2YgdGhlIGVuY2xvc2VkIGB1aVNyZWZgIHN0YXR1cyBpcyB0cnVlKi5cbiAqIEluIG90aGVyIHdvcmRzLCBhbGwgZW5jbG9zZWQgYHVpU3JlZmAgc3RhdHVzZXMgIGFyZSBtZXJnZWQgdG8gYSBzaW5nbGUgc3RhdHVzIHVzaW5nIGB8fGAgKGxvZ2ljYWwgb3IpLlxuICpcbiAqIGBgYGpzXG4gKiA8bGkgKHVpU3JlZlN0YXR1cyk9XCJvblNyZWZTdGF0dXMoJGV2ZW50KVwiIHVpU3JlZj1cImFkbWluXCI+XG4gKiAgIEhvbWVcbiAqICAgPHVsPlxuICogICAgIDxsaT4gPGEgdWlTcmVmPVwiYWRtaW4udXNlcnNcIj5Vc2VyczwvYT4gPC9saT5cbiAqICAgICA8bGk+IDxhIHVpU3JlZj1cImFkbWluLmdyb3Vwc1wiPkdyb3VwczwvYT4gPC9saT5cbiAqICAgPC91bD5cbiAqIDwvbGk+XG4gKiBgYGBcbiAqXG4gKiBJbiB0aGUgYWJvdmUgZXhhbXBsZSwgYCRldmVudC5hY3RpdmUgPT09IHRydWVgIHdoZW4gZWl0aGVyIGBhZG1pbi51c2Vyc2Agb3IgYGFkbWluLmdyb3Vwc2AgaXMgYWN0aXZlLlxuICpcbiAqIC0tLVxuICpcbiAqIFRoaXMgQVBJIGlzIHN1YmplY3QgdG8gY2hhbmdlLlxuICovXG5ARGlyZWN0aXZlKHtcbiAgc2VsZWN0b3I6ICdbdWlTcmVmU3RhdHVzXSxbdWlTcmVmQWN0aXZlXSxbdWlTcmVmQWN0aXZlRXFdJyxcbiAgZXhwb3J0QXM6ICd1aVNyZWZTdGF0dXMnLFxufSlcbmV4cG9ydCBjbGFzcyBVSVNyZWZTdGF0dXMge1xuICAvKiogY3VycmVudCBzdGF0dXNlcyBvZiB0aGUgc3RhdGUvcGFyYW1zIHRoZSB1aVNyZWYgZGlyZWN0aXZlIGlzIGxpbmtpbmcgdG8gKi9cbiAgQE91dHB1dCgndWlTcmVmU3RhdHVzJykgdWlTcmVmU3RhdHVzID0gbmV3IEV2ZW50RW1pdHRlcjxTcmVmU3RhdHVzPihmYWxzZSk7XG4gIC8qKiBNb25pdG9yIGFsbCBjaGlsZCBjb21wb25lbnRzIGZvciBVSVNyZWYocykgKi9cbiAgQENvbnRlbnRDaGlsZHJlbihVSVNyZWYsIHsgZGVzY2VuZGFudHM6IHRydWUgfSlcbiAgcHJpdmF0ZSBfc3JlZnM6IFF1ZXJ5TGlzdDxVSVNyZWY+O1xuXG4gIC8qKiBUaGUgY3VycmVudCBzdGF0dXMgKi9cbiAgc3RhdHVzOiBTcmVmU3RhdHVzO1xuXG4gIC8qKiBAaW50ZXJuYWwgKi8gcHJpdmF0ZSBfc3Vic2NyaXB0aW9uOiBTdWJzY3JpcHRpb247XG4gIC8qKiBAaW50ZXJuYWwgKi8gcHJpdmF0ZSBfc3JlZkNoYW5nZXNTdWI6IFN1YnNjcmlwdGlvbjtcbiAgLyoqIEBpbnRlcm5hbCAqLyBwcml2YXRlIF9zcmVmcyQ6IEJlaGF2aW9yU3ViamVjdDxVSVNyZWZbXT47XG4gIC8qKiBAaW50ZXJuYWwgKi8gcHJpdmF0ZSBfZ2xvYmFsczogVUlSb3V0ZXJHbG9iYWxzO1xuICAvKiogQGludGVybmFsICovIHByaXZhdGUgX2hvc3RVaVNyZWY6IFVJU3JlZjtcbiAgY29uc3RydWN0b3IoQEhvc3QoKSBAU2VsZigpIEBPcHRpb25hbCgpIF9ob3N0VWlTcmVmOiBVSVNyZWYsIF9nbG9iYWxzOiBVSVJvdXRlckdsb2JhbHMpIHtcbiAgICB0aGlzLl9nbG9iYWxzID0gX2dsb2JhbHM7XG4gICAgdGhpcy5faG9zdFVpU3JlZiA9IF9ob3N0VWlTcmVmO1xuICAgIHRoaXMuc3RhdHVzID0gT2JqZWN0LmFzc2lnbih7fSwgaW5hY3RpdmVTdGF0dXMpO1xuICB9XG5cbiAgbmdBZnRlckNvbnRlbnRJbml0KCkge1xuICAgIC8vIE1hcCBlYWNoIHRyYW5zaXRpb24gc3RhcnQgZXZlbnQgdG8gYSBzdHJlYW0gb2Y6XG4gICAgLy8gc3RhcnQgLT4gKHN1Y2Nlc3N8ZXJyb3IpXG4gICAgY29uc3QgdHJhbnNFdmVudHMkOiBPYnNlcnZhYmxlPFRyYW5zRXZ0PiA9IHRoaXMuX2dsb2JhbHMuc3RhcnQkLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHRyYW5zOiBUcmFuc2l0aW9uKSA9PiB7XG4gICAgICAgIGNvbnN0IGV2ZW50ID0gKGV2dDogc3RyaW5nKSA9PiAoeyBldnQsIHRyYW5zIH0gYXMgVHJhbnNFdnQpO1xuXG4gICAgICAgIGNvbnN0IHRyYW5zU3RhcnQkID0gb2YoZXZlbnQoJ3N0YXJ0JykpO1xuICAgICAgICBjb25zdCB0cmFuc1Jlc3VsdCA9IHRyYW5zLnByb21pc2UudGhlbihcbiAgICAgICAgICAoKSA9PiBldmVudCgnc3VjY2VzcycpLFxuICAgICAgICAgICgpID0+IGV2ZW50KCdlcnJvcicpXG4gICAgICAgICk7XG4gICAgICAgIGNvbnN0IHRyYW5zRmluaXNoJCA9IGZyb20odHJhbnNSZXN1bHQpO1xuXG4gICAgICAgIHJldHVybiBjb25jYXQodHJhbnNTdGFydCQsIHRyYW5zRmluaXNoJCk7XG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjb25zdCB3aXRoSG9zdFNyZWYgPSAoY2hpbGRyZW5TcmVmczogVUlTcmVmW10pID0+XG4gICAgICBjaGlsZHJlblNyZWZzLmNvbmNhdCh0aGlzLl9ob3N0VWlTcmVmKS5maWx0ZXIoaWRlbnRpdHkpLnJlZHVjZSh1bmlxUiwgW10pO1xuXG4gICAgLy8gV2F0Y2ggdGhlIEBDb250ZW50Q2hpbGRyZW4gVUlTcmVmW10gY29tcG9uZW50cyBhbmQgZ2V0IHRoZWlyIHRhcmdldCBzdGF0ZXNcbiAgICB0aGlzLl9zcmVmcyQgPSBuZXcgQmVoYXZpb3JTdWJqZWN0KHdpdGhIb3N0U3JlZih0aGlzLl9zcmVmcy50b0FycmF5KCkpKTtcbiAgICB0aGlzLl9zcmVmQ2hhbmdlc1N1YiA9IHRoaXMuX3NyZWZzLmNoYW5nZXMuc3Vic2NyaWJlKChzcmVmczogUXVlcnlMaXN0PFVJU3JlZj4pID0+XG4gICAgICB0aGlzLl9zcmVmcyQubmV4dCh3aXRoSG9zdFNyZWYoc3JlZnMudG9BcnJheSgpKSlcbiAgICApO1xuXG4gICAgY29uc3QgdGFyZ2V0U3RhdGVzJDogT2JzZXJ2YWJsZTxUYXJnZXRTdGF0ZVtdPiA9IHRoaXMuX3NyZWZzJC5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzcmVmczogVUlTcmVmW10pID0+IGNvbWJpbmVMYXRlc3Q8VGFyZ2V0U3RhdGVbXT4oc3JlZnMubWFwKChzcmVmKSA9PiBzcmVmLnRhcmdldFN0YXRlJCkpKVxuICAgICk7XG5cbiAgICAvLyBDYWxjdWxhdGUgdGhlIHN0YXR1cyBvZiBlYWNoIFVJU3JlZiBiYXNlZCBvbiB0aGUgdHJhbnNpdGlvbiBldmVudC5cbiAgICAvLyBSZWR1Y2UgdGhlIHN0YXR1c2VzIChpZiBtdWx0aXBsZSkgYnkgb3ItaW5nIGVhY2ggZmxhZy5cbiAgICB0aGlzLl9zdWJzY3JpcHRpb24gPSB0cmFuc0V2ZW50cyRcbiAgICAgIC5waXBlKFxuICAgICAgICBzd2l0Y2hNYXAoKGV2dDogVHJhbnNFdnQpID0+IHtcbiAgICAgICAgICByZXR1cm4gdGFyZ2V0U3RhdGVzJC5waXBlKFxuICAgICAgICAgICAgbWFwKCh0YXJnZXRzOiBUYXJnZXRTdGF0ZVtdKSA9PiB7XG4gICAgICAgICAgICAgIGNvbnN0IHN0YXR1c2VzOiBTcmVmU3RhdHVzW10gPSB0YXJnZXRzLm1hcCgodGFyZ2V0KSA9PiBnZXRTcmVmU3RhdHVzKGV2dCwgdGFyZ2V0KSk7XG4gICAgICAgICAgICAgIHJldHVybiBzdGF0dXNlcy5yZWR1Y2UobWVyZ2VTcmVmU3RhdHVzKTtcbiAgICAgICAgICAgIH0pXG4gICAgICAgICAgKTtcbiAgICAgICAgfSlcbiAgICAgIClcbiAgICAgIC5zdWJzY3JpYmUodGhpcy5fc2V0U3RhdHVzLmJpbmQodGhpcykpO1xuICB9XG5cbiAgbmdPbkRlc3Ryb3koKSB7XG4gICAgaWYgKHRoaXMuX3N1YnNjcmlwdGlvbikgdGhpcy5fc3Vic2NyaXB0aW9uLnVuc3Vic2NyaWJlKCk7XG4gICAgaWYgKHRoaXMuX3NyZWZDaGFuZ2VzU3ViKSB0aGlzLl9zcmVmQ2hhbmdlc1N1Yi51bnN1YnNjcmliZSgpO1xuICAgIGlmICh0aGlzLl9zcmVmcyQpIHRoaXMuX3NyZWZzJC51bnN1YnNjcmliZSgpO1xuICAgIHRoaXMuX3N1YnNjcmlwdGlvbiA9IHRoaXMuX3NyZWZDaGFuZ2VzU3ViID0gdGhpcy5fc3JlZnMkID0gdW5kZWZpbmVkO1xuICB9XG5cbiAgcHJpdmF0ZSBfc2V0U3RhdHVzKHN0YXR1czogU3JlZlN0YXR1cykge1xuICAgIHRoaXMuc3RhdHVzID0gc3RhdHVzO1xuICAgIHRoaXMudWlTcmVmU3RhdHVzLmVtaXQoc3RhdHVzKTtcbiAgfVxufVxuIl19