import { loadNgModule } from '../lazyLoad/lazyLoadNgModule';
/**
 * This is a [[StateBuilder.builder]] function for ngModule lazy loading in Angular.
 *
 * When the [[StateBuilder]] builds a [[State]] object from a raw [[StateDeclaration]], this builder
 * decorates the `lazyLoad` property for states that have a [[Ng2StateDeclaration.ngModule]] declaration.
 *
 * If the state has a [[Ng2StateDeclaration.ngModule]], it will create a `lazyLoad` function
 * that in turn calls `loadNgModule(loadNgModuleFn)`.
 *
 * #### Example:
 * A state that has a `ngModule`
 * ```js
 * var decl = {
 *   ngModule: () => import('./childModule.ts')
 * }
 * ```
 * would build a state with a `lazyLoad` function like:
 * ```js
 * import { loadNgModule } from "@uirouter/angular";
 * var decl = {
 *   lazyLoad: loadNgModule(() => import('./childModule.ts')
 * }
 * ```
 *
 * If the state has both a `ngModule:` *and* a `lazyLoad`, then the `lazyLoad` is run first.
 *
 * #### Example:
 * ```js
 * var decl = {
 *   lazyLoad: () => import('third-party-library'),
 *   ngModule: () => import('./childModule.ts')
 * }
 * ```
 * would build a state with a `lazyLoad` function like:
 * ```js
 * import { loadNgModule } from "@uirouter/angular";
 * var decl = {
 *   lazyLoad: () => import('third-party-library')
 *       .then(() => loadNgModule(() => import('./childModule.ts'))
 * }
 * ```
 *
 */
export function ng2LazyLoadBuilder(state, parent) {
    const loadNgModuleFn = state['loadChildren'];
    return loadNgModuleFn ? loadNgModule(loadNgModuleFn) : state.lazyLoad;
}
//# sourceMappingURL=data:application/json;base64,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