/**
 * # UI-Router for Angular (v2+)
 *
 * - [@uirouter/angular home page](https://ui-router.github.io/ng2)
 * - [tutorials](https://ui-router.github.io/tutorial/ng2/helloworld)
 * - [quick start repository](http://github.com/ui-router/quickstart-ng2)
 *
 * Getting started:
 *
 * - Use npm. Add a dependency on latest `@uirouter/angular`
 * - Import UI-Router classes directly from `"@uirouter/angular"`
 *
 * ```js
 * import {StateRegistry} from "@uirouter/angular";
 * ```
 *
 * - Create application states (as defined by [[Ng2StateDeclaration]]).
 *
 * ```js
 * export let state1: Ng2StateDeclaration = {
 *   name: 'state1',
 *   component: State1Component,
 *   url: '/one'
 * }
 *
 * export let state2: Ng2StateDeclaration = {
 *   name: 'state2',
 *   component: State2Component,
 *   url: '/two'
 * }
 * ```
 *
 * - Import a [[UIRouterModule.forChild]] module into your feature `NgModule`s.
 *
 * ```js
 * @ NgModule({
 *   imports: [
 *     SharedModule,
 *     UIRouterModule.forChild({ states: [state1, state2 ] })
 *   ],
 *   declarations: [
 *     State1Component,
 *     State2Component,
 *   ]
 * })
 * export class MyFeatureModule {}
 * ```
 *
 * - Import a [[UIRouterModule.forRoot]] module into your application root `NgModule`
 * - Either bootstrap a [[UIView]] component, or add a `<ui-view></ui-view>` viewport to your root component.
 *
 * ```js
 * @ NgModule({
 *   imports: [
 *     BrowserModule,
 *     UIRouterModule.forRoot({ states: [ homeState ] }),
 *     MyFeatureModule,
 *   ],
 *   declarations: [
 *     HomeComponent
 *   ]
 *   bootstrap: [ UIView ]
 * })
 * class RootAppModule {}
 *
 * browserPlatformDynamic.bootstrapModule(RootAppModule);
 * ```
 *
 * - Optionally specify a configuration class [[ChildModule.configClass]] for any module
 * to perform any router configuration during bootstrap or lazyload.
 * Pass the class to [[UIRouterModule.forRoot]] or [[UIRouterModule.forChild]].
 *
 * ```js
 * import {UIRouter} from "@uirouter/angular";
 *
 * @ Injectable()
 * export class MyUIRouterConfig {
 *   // Constructor is injectable
 *   constructor(uiRouter: UIRouter) {
 *     uiRouter.urlMatcherFactory.type('datetime', myDateTimeParamType);
 *   }
 * }
 * ```
 */
import { Injector, PLATFORM_ID, APP_INITIALIZER } from '@angular/core';
import { isPlatformBrowser } from '@angular/common';
import { UIRouter, StateRegistry, StateService, TransitionService, UrlMatcherFactory, UrlRouter, ViewService, UrlService, UIRouterGlobals, services, Resolvable, NATIVE_INJECTOR_TOKEN, } from '@uirouter/core';
import { UIView } from './directives/uiView';
import { UIROUTER_MODULE_TOKEN, UIROUTER_ROOT_MODULE } from './injectionTokens';
import { ng2ViewsBuilder, Ng2ViewConfig } from './statebuilders/views';
import { applyRootModuleConfig, applyModuleConfig } from './uiRouterConfig';
import { servicesPlugin } from '@uirouter/core';
import { ng2LazyLoadBuilder } from './statebuilders/lazyLoad';
import { UIRouterRx } from '@uirouter/rx';
import { LocationStrategy } from '@angular/common';
import { Ng2LocationServices } from './location/locationService';
import { Ng2LocationConfig } from './location/locationConfig';
/**
 * This is a factory function for a UIRouter instance
 *
 * Creates a UIRouter instance and configures it for Angular, then invokes router bootstrap.
 * This function is used as an Angular `useFactory` Provider.
 */
export function uiRouterFactory(locationStrategy, rootModules, modules, injector) {
    if (rootModules.length !== 1) {
        throw new Error("Exactly one UIRouterModule.forRoot() should be in the bootstrapped app module's imports: []");
    }
    // ----------------- Create router -----------------
    // Create a new ng2 UIRouter and configure it for ng2
    const router = new UIRouter();
    // Add RxJS plugin
    router.plugin(UIRouterRx);
    // Add $q-like and $injector-like service APIs
    router.plugin(servicesPlugin);
    // ----------------- Monkey Patches ----------------
    // Monkey patch the services.$injector to use the root ng2 Injector
    services.$injector.get = injector.get.bind(injector);
    // ----------------- Configure for ng2 -------------
    router.locationService = new Ng2LocationServices(router, locationStrategy, isPlatformBrowser(injector.get(PLATFORM_ID)));
    router.locationConfig = new Ng2LocationConfig(router, locationStrategy);
    // Apply ng2 ui-view handling code
    const viewConfigFactory = (path, config) => new Ng2ViewConfig(path, config);
    router.viewService._pluginapi._viewConfigFactory('ng2', viewConfigFactory);
    // Apply statebuilder decorator for ng2 NgModule registration
    const registry = router.stateRegistry;
    registry.decorator('views', ng2ViewsBuilder);
    registry.decorator('lazyLoad', ng2LazyLoadBuilder);
    // Prep the tree of NgModule by placing the root NgModule's Injector on the root state.
    const ng2InjectorResolvable = Resolvable.fromData(NATIVE_INJECTOR_TOKEN, injector);
    registry.root().resolvables.push(ng2InjectorResolvable);
    // Auto-flush the parameter type queue
    router.urlMatcherFactory.$get();
    // ----------------- Initialize router -------------
    rootModules.forEach((moduleConfig) => applyRootModuleConfig(router, injector, moduleConfig));
    modules.forEach((moduleConfig) => applyModuleConfig(router, injector, moduleConfig));
    return router;
}
// Start monitoring the URL when the app starts
export function appInitializer(router) {
    return () => {
        if (!router.urlRouter.interceptDeferred) {
            router.urlService.listen();
            router.urlService.sync();
        }
    };
}
export function parentUIViewInjectFactory(r) {
    return { fqn: null, context: r.root() };
}
export const _UIROUTER_INSTANCE_PROVIDERS = [
    {
        provide: UIRouter,
        useFactory: uiRouterFactory,
        deps: [LocationStrategy, UIROUTER_ROOT_MODULE, UIROUTER_MODULE_TOKEN, Injector],
    },
    { provide: UIView.PARENT_INJECT, useFactory: parentUIViewInjectFactory, deps: [StateRegistry] },
    { provide: APP_INITIALIZER, useFactory: appInitializer, deps: [UIRouter], multi: true },
];
export function fnStateService(r) {
    return r.stateService;
}
export function fnTransitionService(r) {
    return r.transitionService;
}
export function fnUrlMatcherFactory(r) {
    return r.urlMatcherFactory;
}
export function fnUrlRouter(r) {
    return r.urlRouter;
}
export function fnUrlService(r) {
    return r.urlService;
}
export function fnViewService(r) {
    return r.viewService;
}
export function fnStateRegistry(r) {
    return r.stateRegistry;
}
export function fnGlobals(r) {
    return r.globals;
}
export const _UIROUTER_SERVICE_PROVIDERS = [
    { provide: StateService, useFactory: fnStateService, deps: [UIRouter] },
    { provide: TransitionService, useFactory: fnTransitionService, deps: [UIRouter] },
    { provide: UrlMatcherFactory, useFactory: fnUrlMatcherFactory, deps: [UIRouter] },
    { provide: UrlRouter, useFactory: fnUrlRouter, deps: [UIRouter] },
    { provide: UrlService, useFactory: fnUrlService, deps: [UIRouter] },
    { provide: ViewService, useFactory: fnViewService, deps: [UIRouter] },
    { provide: StateRegistry, useFactory: fnStateRegistry, deps: [UIRouter] },
    { provide: UIRouterGlobals, useFactory: fnGlobals, deps: [UIRouter] },
];
/**
 * The UI-Router providers, for use in your application bootstrap
 *
 * @deprecated use [[UIRouterModule.forRoot]]
 */
export const UIROUTER_PROVIDERS = _UIROUTER_INSTANCE_PROVIDERS.concat(_UIROUTER_SERVICE_PROVIDERS);
//# sourceMappingURL=data:application/json;base64,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