import { NgModuleFactory, Compiler, NgModuleFactoryLoader } from '@angular/core';
import { UIRouter, Resolvable, NATIVE_INJECTOR_TOKEN, isString, unnestR, inArray, uniqR, } from '@uirouter/core';
import { UIROUTER_MODULE_TOKEN, UIROUTER_ROOT_MODULE } from '../injectionTokens';
import { applyModuleConfig } from '../uiRouterConfig';
/**
 * Returns a function which lazy loads a nested module
 *
 * This is primarily used by the [[ng2LazyLoadBuilder]] when processing [[Ng2StateDeclaration.loadChildren]].
 *
 * It could also be used manually as a [[StateDeclaration.lazyLoad]] property to lazy load an `NgModule` and its state(s).
 *
 * #### Example:
 * Using `import()` and named export of `HomeModule`
 * ```js
 * declare var System;
 * var futureState = {
 *   name: 'home.**',
 *   url: '/home',
 *   lazyLoad: loadNgModule(() => import('./home/home.module').then(result => result.HomeModule))
 * }
 * ```
 *
 * #### Example:
 * Using a path (string) to the module
 * ```js
 * var futureState = {
 *   name: 'home.**',
 *   url: '/home',
 *   lazyLoad: loadNgModule('./home/home.module#HomeModule')
 * }
 * ```
 *
 *
 * @param moduleToLoad a path (string) to the NgModule to load.
 *    Or a function which loads the NgModule code which should
 *    return a reference to  the `NgModule` class being loaded (or a `Promise` for it).
 *
 * @returns A function which takes a transition, which:
 * - Gets the Injector (scoped properly for the destination state)
 * - Loads and creates the NgModule
 * - Finds the "replacement state" for the target state, and adds the new NgModule Injector to it (as a resolve)
 * - Returns the new states array
 */
export function loadNgModule(moduleToLoad) {
    return (transition, stateObject) => {
        const ng2Injector = transition.injector().get(NATIVE_INJECTOR_TOKEN);
        const createModule = (factory) => factory.create(ng2Injector);
        const applyModule = (moduleRef) => applyNgModule(transition, moduleRef, ng2Injector, stateObject);
        return loadModuleFactory(moduleToLoad, ng2Injector).then(createModule).then(applyModule);
    };
}
/**
 * Returns the module factory that can be used to instantiate a module
 *
 * For strings this:
 * - Finds the correct NgModuleFactoryLoader
 * - Loads the new NgModuleFactory from the path string (async)
 *
 * For a Type<any> or Promise<Type<any>> this:
 * - Compiles the component type (if not running with AOT)
 * - Returns the NgModuleFactory resulting from compilation (or direct loading if using AOT) as a Promise
 *
 * @internal
 */
export function loadModuleFactory(moduleToLoad, ng2Injector) {
    if (isString(moduleToLoad)) {
        return ng2Injector.get(NgModuleFactoryLoader).load(moduleToLoad);
    }
    const compiler = ng2Injector.get(Compiler);
    const unwrapEsModuleDefault = (x) => (x && x.__esModule && x['default'] ? x['default'] : x);
    return Promise.resolve(moduleToLoad())
        .then(unwrapEsModuleDefault)
        .then((t) => {
        if (t instanceof NgModuleFactory) {
            return t;
        }
        return compiler.compileModuleAsync(t);
    });
}
/**
 * Apply the UI-Router Modules found in the lazy loaded module.
 *
 * Apply the Lazy Loaded NgModule's newly created Injector to the right state in the state tree.
 *
 * Lazy loading uses a placeholder state which is removed (and replaced) after the module is loaded.
 * The NgModule should include a state with the same name as the placeholder.
 *
 * Find the *newly loaded state* with the same name as the *placeholder state*.
 * The NgModule's Injector (and ComponentFactoryResolver) will be added to that state.
 * The Injector/Factory are used when creating Components for the `replacement` state and all its children.
 *
 * @internal
 */
export function applyNgModule(transition, ng2Module, parentInjector, lazyLoadState) {
    const injector = ng2Module.injector;
    const uiRouter = injector.get(UIRouter);
    const registry = uiRouter.stateRegistry;
    const originalName = lazyLoadState.name;
    const originalState = registry.get(originalName);
    // Check if it's a future state (ends with .**)
    const isFuture = /^(.*)\.\*\*$/.exec(originalName);
    // Final name (without the .**)
    const replacementName = isFuture && isFuture[1];
    const newRootModules = multiProviderParentChildDelta(parentInjector, injector, UIROUTER_ROOT_MODULE).reduce(uniqR, []);
    const newChildModules = multiProviderParentChildDelta(parentInjector, injector, UIROUTER_MODULE_TOKEN).reduce(uniqR, []);
    if (newRootModules.length) {
        console.log(newRootModules); // tslint:disable-line:no-console
        throw new Error('Lazy loaded modules should not contain a UIRouterModule.forRoot() module');
    }
    const newStateObjects = newChildModules
        .map((module) => applyModuleConfig(uiRouter, injector, module))
        .reduce(unnestR, [])
        .reduce(uniqR, []);
    if (isFuture) {
        const replacementState = registry.get(replacementName);
        if (!replacementState || replacementState === originalState) {
            throw new Error(`The Future State named '${originalName}' lazy loaded an NgModule. ` +
                `The lazy loaded NgModule must have a state named '${replacementName}' ` +
                `which replaces the (placeholder) '${originalName}' Future State. ` +
                `Add a '${replacementName}' state to the lazy loaded NgModule ` +
                `using UIRouterModule.forChild({ states: CHILD_STATES }).`);
        }
    }
    // Supply the newly loaded states with the Injector from the lazy loaded NgModule.
    // If a tree of states is lazy loaded, only add the injector to the root of the lazy loaded tree.
    // The children will get the injector by resolve inheritance.
    const newParentStates = newStateObjects.filter((state) => !inArray(newStateObjects, state.parent));
    // Add the Injector to the top of the lazy loaded state tree as a resolve
    newParentStates.forEach((state) => state.resolvables.push(Resolvable.fromData(NATIVE_INJECTOR_TOKEN, injector)));
    return {};
}
/**
 * Returns the new dependency injection values from the Child Injector
 *
 * When a DI token is defined as multi: true, the child injector
 * can add new values for the token.
 *
 * This function returns the values added by the child injector,  and excludes all values from the parent injector.
 *
 * @internal
 */
export function multiProviderParentChildDelta(parent, child, token) {
    const childVals = child.get(token, []);
    const parentVals = parent.get(token, []);
    return childVals.filter((val) => parentVals.indexOf(val) === -1);
}
//# sourceMappingURL=data:application/json;base64,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