import { __decorate, __param } from "tslib";
import { Directive, Output, EventEmitter, ContentChildren, Host, Self, Optional } from '@angular/core';
import { UISref } from './uiSref';
import { PathNode, Transition, TargetState, StateObject, anyTrueR, tail, unnestR, Predicate, UIRouterGlobals, Param, PathUtils, identity, uniqR, } from '@uirouter/core';
import { BehaviorSubject, of, from, combineLatest, concat } from 'rxjs';
import { switchMap, map } from 'rxjs/operators';
/** @internal */
const inactiveStatus = {
    active: false,
    exact: false,
    entering: false,
    exiting: false,
    targetStates: [],
};
/**
 * Returns a Predicate<PathNode[]>
 *
 * The predicate returns true when the target state (and param values)
 * match the (tail of) the path, and the path's param values
 *
 * @internal
 */
const pathMatches = (target) => {
    if (!target.exists())
        return () => false;
    const state = target.$state();
    const targetParamVals = target.params();
    const targetPath = PathUtils.buildPath(target);
    const paramSchema = targetPath
        .map((node) => node.paramSchema)
        .reduce(unnestR, [])
        .filter((param) => targetParamVals.hasOwnProperty(param.id));
    return (path) => {
        const tailNode = tail(path);
        if (!tailNode || tailNode.state !== state)
            return false;
        const paramValues = PathUtils.paramValues(path);
        return Param.equals(paramSchema, paramValues, targetParamVals);
    };
};
const ɵ0 = pathMatches;
/**
 * Given basePath: [a, b], appendPath: [c, d]),
 * Expands the path to [c], [c, d]
 * Then appends each to [a,b,] and returns: [a, b, c], [a, b, c, d]
 *
 * @internal
 */
function spreadToSubPaths(basePath, appendPath) {
    return appendPath.map((node) => basePath.concat(PathUtils.subPath(appendPath, (n) => n.state === node.state)));
}
/**
 * Given a TransEvt (Transition event: started, success, error)
 * and a UISref Target State, return a SrefStatus object
 * which represents the current status of that Sref:
 * active, activeEq (exact match), entering, exiting
 *
 * @internal
 */
function getSrefStatus(event, srefTarget) {
    const pathMatchesTarget = pathMatches(srefTarget);
    const tc = event.trans.treeChanges();
    const isStartEvent = event.evt === 'start';
    const isSuccessEvent = event.evt === 'success';
    const activePath = isSuccessEvent ? tc.to : tc.from;
    const isActive = () => spreadToSubPaths([], activePath).map(pathMatchesTarget).reduce(anyTrueR, false);
    const isExact = () => pathMatchesTarget(activePath);
    const isEntering = () => spreadToSubPaths(tc.retained, tc.entering).map(pathMatchesTarget).reduce(anyTrueR, false);
    const isExiting = () => spreadToSubPaths(tc.retained, tc.exiting).map(pathMatchesTarget).reduce(anyTrueR, false);
    return {
        active: isActive(),
        exact: isExact(),
        entering: isStartEvent ? isEntering() : false,
        exiting: isStartEvent ? isExiting() : false,
        targetStates: [srefTarget],
    };
}
/** @internal */
function mergeSrefStatus(left, right) {
    return {
        active: left.active || right.active,
        exact: left.exact || right.exact,
        entering: left.entering || right.entering,
        exiting: left.exiting || right.exiting,
        targetStates: left.targetStates.concat(right.targetStates),
    };
}
/**
 * A directive which emits events when a paired [[UISref]] status changes.
 *
 * This directive is primarily used by the [[UISrefActive]] directives to monitor `UISref`(s).
 *
 * This directive shares two attribute selectors with `UISrefActive`:
 *
 * - `[uiSrefActive]`
 * - `[uiSrefActiveEq]`.
 *
 * Thus, whenever a `UISrefActive` directive is created, a `UISrefStatus` directive is also created.
 *
 * Most apps should simply use `UISrefActive`, but some advanced components may want to process the
 * [[SrefStatus]] events directly.
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatusChanged($event)">
 *   <a uiSref="book" [uiParams]="{ bookId: book.id }">Book {{ book.name }}</a>
 * </li>
 * ```
 *
 * The `uiSrefStatus` event is emitted whenever an enclosed `uiSref`'s status changes.
 * The event emitted is of type [[SrefStatus]], and has boolean values for `active`, `exact`, `entering`, and `exiting`; also has a [[StateOrName]] `identifier`value.
 *
 * The values from this event can be captured and stored on a component (then applied, e.g., using ngClass).
 *
 * ---
 *
 * A single `uiSrefStatus` can enclose multiple `uiSref`.
 * Each status boolean (`active`, `exact`, `entering`, `exiting`) will be true if *any of the enclosed `uiSref` status is true*.
 * In other words, all enclosed `uiSref` statuses  are merged to a single status using `||` (logical or).
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatus($event)" uiSref="admin">
 *   Home
 *   <ul>
 *     <li> <a uiSref="admin.users">Users</a> </li>
 *     <li> <a uiSref="admin.groups">Groups</a> </li>
 *   </ul>
 * </li>
 * ```
 *
 * In the above example, `$event.active === true` when either `admin.users` or `admin.groups` is active.
 *
 * ---
 *
 * This API is subject to change.
 */
let UISrefStatus = class UISrefStatus {
    constructor(_hostUiSref, _globals) {
        /** current statuses of the state/params the uiSref directive is linking to */
        this.uiSrefStatus = new EventEmitter(false);
        this._globals = _globals;
        this._hostUiSref = _hostUiSref;
        this.status = Object.assign({}, inactiveStatus);
    }
    ngAfterContentInit() {
        // Map each transition start event to a stream of:
        // start -> (success|error)
        const transEvents$ = this._globals.start$.pipe(switchMap((trans) => {
            const event = (evt) => ({ evt, trans });
            const transStart$ = of(event('start'));
            const transResult = trans.promise.then(() => event('success'), () => event('error'));
            const transFinish$ = from(transResult);
            return concat(transStart$, transFinish$);
        }));
        const withHostSref = (childrenSrefs) => childrenSrefs.concat(this._hostUiSref).filter(identity).reduce(uniqR, []);
        // Watch the @ContentChildren UISref[] components and get their target states
        this._srefs$ = new BehaviorSubject(withHostSref(this._srefs.toArray()));
        this._srefChangesSub = this._srefs.changes.subscribe((srefs) => this._srefs$.next(withHostSref(srefs.toArray())));
        const targetStates$ = this._srefs$.pipe(switchMap((srefs) => combineLatest(srefs.map((sref) => sref.targetState$))));
        // Calculate the status of each UISref based on the transition event.
        // Reduce the statuses (if multiple) by or-ing each flag.
        this._subscription = transEvents$
            .pipe(switchMap((evt) => {
            return targetStates$.pipe(map((targets) => {
                const statuses = targets.map((target) => getSrefStatus(evt, target));
                return statuses.reduce(mergeSrefStatus);
            }));
        }))
            .subscribe(this._setStatus.bind(this));
    }
    ngOnDestroy() {
        if (this._subscription)
            this._subscription.unsubscribe();
        if (this._srefChangesSub)
            this._srefChangesSub.unsubscribe();
        if (this._srefs$)
            this._srefs$.unsubscribe();
        this._subscription = this._srefChangesSub = this._srefs$ = undefined;
    }
    _setStatus(status) {
        this.status = status;
        this.uiSrefStatus.emit(status);
    }
};
UISrefStatus.ctorParameters = () => [
    { type: UISref, decorators: [{ type: Host }, { type: Self }, { type: Optional }] },
    { type: UIRouterGlobals }
];
__decorate([
    Output('uiSrefStatus')
], UISrefStatus.prototype, "uiSrefStatus", void 0);
__decorate([
    ContentChildren(UISref, { descendants: true })
], UISrefStatus.prototype, "_srefs", void 0);
UISrefStatus = __decorate([
    Directive({
        selector: '[uiSrefStatus],[uiSrefActive],[uiSrefActiveEq]',
        exportAs: 'uiSrefStatus',
    }),
    __param(0, Host()), __param(0, Self()), __param(0, Optional())
], UISrefStatus);
export { UISrefStatus };
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,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