import { EMPTY, fromEvent, of, race, Subject, timer } from 'rxjs';
import { endWith, filter, takeUntil } from 'rxjs/operators';
import { getTransitionDurationMs } from './util';
import { environment } from '../../environment';
import { runInZone } from '../util';
const noopFn = () => { };
const ɵ0 = noopFn;
const { transitionTimerDelayMs } = environment;
const runningTransitions = new Map();
export const ngbRunTransition = (zone, element, startFn, options) => {
    // Getting initial context from options
    let context = options.context || {};
    // Checking if there are already running transitions on the given element.
    const running = runningTransitions.get(element);
    if (running) {
        switch (options.runningTransition) {
            // If there is one running and we want for it to 'continue' to run, we have to cancel the new one.
            // We're not emitting any values, but simply completing the observable (EMPTY).
            case 'continue':
                return EMPTY;
            // If there is one running and we want for it to 'stop', we have to complete the running one.
            // We're simply completing the running one and not emitting any values and merging newly provided context
            // with the one coming from currently running transition.
            case 'stop':
                zone.run(() => running.transition$.complete());
                context = Object.assign(running.context, context);
                runningTransitions.delete(element);
        }
    }
    // Running the start function
    const endFn = startFn(element, options.animation, context) || noopFn;
    // If 'prefer-reduced-motion' is enabled, the 'transition' will be set to 'none'.
    // If animations are disabled, we have to emit a value and complete the observable
    // In this case we have to call the end function, but can finish immediately by emitting a value,
    // completing the observable and executing end functions synchronously.
    if (!options.animation || window.getComputedStyle(element).transitionProperty === 'none') {
        zone.run(() => endFn());
        return of(undefined).pipe(runInZone(zone));
    }
    // Starting a new transition
    const transition$ = new Subject();
    const finishTransition$ = new Subject();
    const stop$ = transition$.pipe(endWith(true));
    runningTransitions.set(element, {
        transition$,
        complete: () => {
            finishTransition$.next();
            finishTransition$.complete();
        },
        context
    });
    const transitionDurationMs = getTransitionDurationMs(element);
    // 1. We have to both listen for the 'transitionend' event and have a 'just-in-case' timer,
    // because 'transitionend' event might not be fired in some browsers, if the transitioning
    // element becomes invisible (ex. when scrolling, making browser tab inactive, etc.). The timer
    // guarantees, that we'll release the DOM element and complete 'ngbRunTransition'.
    // 2. We need to filter transition end events, because they might bubble from shorter transitions
    // on inner DOM elements. We're only interested in the transition on the 'element' itself.
    zone.runOutsideAngular(() => {
        const transitionEnd$ = fromEvent(element, 'transitionend').pipe(takeUntil(stop$), filter(({ target }) => target === element));
        const timer$ = timer(transitionDurationMs + transitionTimerDelayMs).pipe(takeUntil(stop$));
        race(timer$, transitionEnd$, finishTransition$).pipe(takeUntil(stop$)).subscribe(() => {
            runningTransitions.delete(element);
            zone.run(() => {
                endFn();
                transition$.next();
                transition$.complete();
            });
        });
    });
    return transition$.asObservable();
};
export const ngbCompleteTransition = (element) => {
    var _a;
    (_a = runningTransitions.get(element)) === null || _a === void 0 ? void 0 : _a.complete();
};
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,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