import { ChangeDetectionStrategy, ChangeDetectorRef, Component, Directive, ElementRef, Input, NgZone, ViewChildren, ViewEncapsulation } from '@angular/core';
import { distinctUntilChanged, skip, startWith, takeUntil } from 'rxjs/operators';
import { ngbNavFadeInTransition, ngbNavFadeOutTransition } from './nav-transition';
import { ngbRunTransition } from '../util/transition/ngbTransition';
export class NgbNavPane {
    constructor(elRef) {
        this.elRef = elRef;
    }
}
NgbNavPane.decorators = [
    { type: Directive, args: [{
                selector: '[ngbNavPane]',
                host: {
                    '[id]': 'item.panelDomId',
                    'class': 'tab-pane',
                    '[class.fade]': 'nav.animation',
                    '[attr.role]': 'role ? role : nav.roles ? "tabpanel" : undefined',
                    '[attr.aria-labelledby]': 'item.domId'
                }
            },] }
];
NgbNavPane.ctorParameters = () => [
    { type: ElementRef }
];
NgbNavPane.propDecorators = {
    item: [{ type: Input }],
    nav: [{ type: Input }],
    role: [{ type: Input }]
};
/**
 * The outlet where currently active nav content will be displayed.
 *
 * @since 5.2.0
 */
export class NgbNavOutlet {
    constructor(_cd, _ngZone) {
        this._cd = _cd;
        this._ngZone = _ngZone;
        this._activePane = null;
    }
    isPanelTransitioning(item) { var _a; return ((_a = this._activePane) === null || _a === void 0 ? void 0 : _a.item) === item; }
    ngAfterViewInit() {
        var _a;
        // initial display
        this._updateActivePane();
        // this will be emitted for all 3 types of nav changes: .select(), [activeId] or (click)
        this.nav.navItemChange$
            .pipe(takeUntil(this.nav.destroy$), startWith(((_a = this._activePane) === null || _a === void 0 ? void 0 : _a.item) || null), distinctUntilChanged(), skip(1))
            .subscribe(nextItem => {
            const options = { animation: this.nav.animation, runningTransition: 'stop' };
            // next panel we're switching to will only appear in DOM after the change detection is done
            // and `this._panes` will be updated
            this._cd.detectChanges();
            // fading out
            if (this._activePane) {
                ngbRunTransition(this._ngZone, this._activePane.elRef.nativeElement, ngbNavFadeOutTransition, options)
                    .subscribe(() => {
                    var _a;
                    const activeItem = (_a = this._activePane) === null || _a === void 0 ? void 0 : _a.item;
                    this._activePane = this._getPaneForItem(nextItem);
                    // mark for check when transition finishes as outlet or parent containers might be OnPush
                    // without this the panes that have "faded out" will stay in DOM
                    this._cd.markForCheck();
                    // fading in
                    if (this._activePane) {
                        // we have to add the '.active' class before running the transition,
                        // because it should be in place before `ngbRunTransition` does `reflow()`
                        this._activePane.elRef.nativeElement.classList.add('active');
                        ngbRunTransition(this._ngZone, this._activePane.elRef.nativeElement, ngbNavFadeInTransition, options)
                            .subscribe(() => {
                            if (nextItem) {
                                nextItem.shown.emit();
                                this.nav.shown.emit(nextItem.id);
                            }
                        });
                    }
                    if (activeItem) {
                        activeItem.hidden.emit();
                        this.nav.hidden.emit(activeItem.id);
                    }
                });
            }
            else {
                this._updateActivePane();
            }
        });
    }
    _updateActivePane() {
        var _a, _b;
        this._activePane = this._getActivePane();
        (_a = this._activePane) === null || _a === void 0 ? void 0 : _a.elRef.nativeElement.classList.add('show');
        (_b = this._activePane) === null || _b === void 0 ? void 0 : _b.elRef.nativeElement.classList.add('active');
    }
    _getPaneForItem(item) {
        return this._panes && this._panes.find(pane => pane.item === item) || null;
    }
    _getActivePane() {
        return this._panes && this._panes.find(pane => pane.item.active) || null;
    }
}
NgbNavOutlet.decorators = [
    { type: Component, args: [{
                selector: '[ngbNavOutlet]',
                host: { '[class.tab-content]': 'true' },
                encapsulation: ViewEncapsulation.None,
                changeDetection: ChangeDetectionStrategy.OnPush,
                template: `
    <ng-template ngFor let-item [ngForOf]="nav.items">
      <div ngbNavPane *ngIf="item.isPanelInDom() || isPanelTransitioning(item)" [item]="item" [nav]="nav" [role]="paneRole">
        <ng-template [ngTemplateOutlet]="item.contentTpl?.templateRef || null"
                     [ngTemplateOutletContext]="{$implicit: item.active || isPanelTransitioning(item)}"></ng-template>
      </div>
    </ng-template>
  `
            },] }
];
NgbNavOutlet.ctorParameters = () => [
    { type: ChangeDetectorRef },
    { type: NgZone }
];
NgbNavOutlet.propDecorators = {
    _panes: [{ type: ViewChildren, args: [NgbNavPane,] }],
    paneRole: [{ type: Input }],
    nav: [{ type: Input, args: ['ngbNavOutlet',] }]
};
//# sourceMappingURL=data:application/json;base64,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