import { NgbDate } from '../ngb-date';
const PARTS_PER_HOUR = 1080;
const PARTS_PER_DAY = 24 * PARTS_PER_HOUR;
const PARTS_FRACTIONAL_MONTH = 12 * PARTS_PER_HOUR + 793;
const PARTS_PER_MONTH = 29 * PARTS_PER_DAY + PARTS_FRACTIONAL_MONTH;
const BAHARAD = 11 * PARTS_PER_HOUR + 204;
const HEBREW_DAY_ON_JAN_1_1970 = 2092591;
const GREGORIAN_EPOCH = 1721425.5;
function isGregorianLeapYear(year) {
    return year % 4 === 0 && year % 100 !== 0 || year % 400 === 0;
}
function numberOfFirstDayInYear(year) {
    let monthsBeforeYear = Math.floor((235 * year - 234) / 19);
    let fractionalMonthsBeforeYear = monthsBeforeYear * PARTS_FRACTIONAL_MONTH + BAHARAD;
    let dayNumber = monthsBeforeYear * 29 + Math.floor(fractionalMonthsBeforeYear / PARTS_PER_DAY);
    let timeOfDay = fractionalMonthsBeforeYear % PARTS_PER_DAY;
    let dayOfWeek = dayNumber % 7; // 0 == Monday
    if (dayOfWeek === 2 || dayOfWeek === 4 || dayOfWeek === 6) {
        dayNumber++;
        dayOfWeek = dayNumber % 7;
    }
    if (dayOfWeek === 1 && timeOfDay > 15 * PARTS_PER_HOUR + 204 && !isHebrewLeapYear(year)) {
        dayNumber += 2;
    }
    else if (dayOfWeek === 0 && timeOfDay > 21 * PARTS_PER_HOUR + 589 && isHebrewLeapYear(year - 1)) {
        dayNumber++;
    }
    return dayNumber;
}
function getDaysInGregorianMonth(month, year) {
    let days = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    if (isGregorianLeapYear(year)) {
        days[1]++;
    }
    return days[month - 1];
}
function getHebrewMonths(year) {
    return isHebrewLeapYear(year) ? 13 : 12;
}
/**
 * Returns the number of days in a specific Hebrew year.
 * `year` is any Hebrew year.
 */
function getDaysInHebrewYear(year) {
    return numberOfFirstDayInYear(year + 1) - numberOfFirstDayInYear(year);
}
export function isHebrewLeapYear(year) {
    if (year != null) {
        let b = (year * 12 + 17) % 19;
        return b >= ((b < 0) ? -7 : 12);
    }
    return false;
}
/**
 * Returns the number of days in a specific Hebrew month.
 * `month` is 1 for Nisan, 2 for Iyar etc. Note: Hebrew leap year contains 13 months.
 * `year` is any Hebrew year.
 */
export function getDaysInHebrewMonth(month, year) {
    let yearLength = numberOfFirstDayInYear(year + 1) - numberOfFirstDayInYear(year);
    let yearType = (yearLength <= 380 ? yearLength : (yearLength - 30)) - 353;
    let leapYear = isHebrewLeapYear(year);
    let daysInMonth = leapYear ? [30, 29, 29, 29, 30, 30, 29, 30, 29, 30, 29, 30, 29] :
        [30, 29, 29, 29, 30, 29, 30, 29, 30, 29, 30, 29];
    if (yearType > 0) {
        daysInMonth[2]++; // Kislev gets an extra day in normal or complete years.
    }
    if (yearType > 1) {
        daysInMonth[1]++; // Heshvan gets an extra day in complete years only.
    }
    return daysInMonth[month - 1];
}
export function getDayNumberInHebrewYear(date) {
    let numberOfDay = 0;
    for (let i = 1; i < date.month; i++) {
        numberOfDay += getDaysInHebrewMonth(i, date.year);
    }
    return numberOfDay + date.day;
}
export function setHebrewMonth(date, val) {
    let after = val >= 0;
    if (!after) {
        val = -val;
    }
    while (val > 0) {
        if (after) {
            if (val > getHebrewMonths(date.year) - date.month) {
                val -= getHebrewMonths(date.year) - date.month + 1;
                date.year++;
                date.month = 1;
            }
            else {
                date.month += val;
                val = 0;
            }
        }
        else {
            if (val >= date.month) {
                date.year--;
                val -= date.month;
                date.month = getHebrewMonths(date.year);
            }
            else {
                date.month -= val;
                val = 0;
            }
        }
    }
    return date;
}
export function setHebrewDay(date, val) {
    let after = val >= 0;
    if (!after) {
        val = -val;
    }
    while (val > 0) {
        if (after) {
            if (val > getDaysInHebrewYear(date.year) - getDayNumberInHebrewYear(date)) {
                val -= getDaysInHebrewYear(date.year) - getDayNumberInHebrewYear(date) + 1;
                date.year++;
                date.month = 1;
                date.day = 1;
            }
            else if (val > getDaysInHebrewMonth(date.month, date.year) - date.day) {
                val -= getDaysInHebrewMonth(date.month, date.year) - date.day + 1;
                date.month++;
                date.day = 1;
            }
            else {
                date.day += val;
                val = 0;
            }
        }
        else {
            if (val >= date.day) {
                val -= date.day;
                date.month--;
                if (date.month === 0) {
                    date.year--;
                    date.month = getHebrewMonths(date.year);
                }
                date.day = getDaysInHebrewMonth(date.month, date.year);
            }
            else {
                date.day -= val;
                val = 0;
            }
        }
    }
    return date;
}
/**
 * Returns the equivalent Hebrew date value for a give input Gregorian date.
 * `gdate` is a JS Date to be converted to Hebrew date.
 */
export function fromGregorian(gdate) {
    const date = new Date(gdate);
    const gYear = date.getFullYear(), gMonth = date.getMonth(), gDay = date.getDate();
    let julianDay = GREGORIAN_EPOCH - 1 + 365 * (gYear - 1) + Math.floor((gYear - 1) / 4) -
        Math.floor((gYear - 1) / 100) + Math.floor((gYear - 1) / 400) +
        Math.floor((367 * (gMonth + 1) - 362) / 12 + (gMonth + 1 <= 2 ? 0 : isGregorianLeapYear(gYear) ? -1 : -2) + gDay);
    julianDay = Math.floor(julianDay + 0.5);
    let daysSinceHebEpoch = julianDay - 347997;
    let monthsSinceHebEpoch = Math.floor(daysSinceHebEpoch * PARTS_PER_DAY / PARTS_PER_MONTH);
    let hYear = Math.floor((monthsSinceHebEpoch * 19 + 234) / 235) + 1;
    let firstDayOfThisYear = numberOfFirstDayInYear(hYear);
    let dayOfYear = daysSinceHebEpoch - firstDayOfThisYear;
    while (dayOfYear < 1) {
        hYear--;
        firstDayOfThisYear = numberOfFirstDayInYear(hYear);
        dayOfYear = daysSinceHebEpoch - firstDayOfThisYear;
    }
    let hMonth = 1;
    let hDay = dayOfYear;
    while (hDay > getDaysInHebrewMonth(hMonth, hYear)) {
        hDay -= getDaysInHebrewMonth(hMonth, hYear);
        hMonth++;
    }
    return new NgbDate(hYear, hMonth, hDay);
}
/**
 * Returns the equivalent JS date value for a given Hebrew date.
 * `hebrewDate` is an Hebrew date to be converted to Gregorian.
 */
export function toGregorian(hebrewDate) {
    const hYear = hebrewDate.year;
    const hMonth = hebrewDate.month;
    const hDay = hebrewDate.day;
    let days = numberOfFirstDayInYear(hYear);
    for (let i = 1; i < hMonth; i++) {
        days += getDaysInHebrewMonth(i, hYear);
    }
    days += hDay;
    let diffDays = days - HEBREW_DAY_ON_JAN_1_1970;
    let after = diffDays >= 0;
    if (!after) {
        diffDays = -diffDays;
    }
    let gYear = 1970;
    let gMonth = 1;
    let gDay = 1;
    while (diffDays > 0) {
        if (after) {
            if (diffDays >= (isGregorianLeapYear(gYear) ? 366 : 365)) {
                diffDays -= isGregorianLeapYear(gYear) ? 366 : 365;
                gYear++;
            }
            else if (diffDays >= getDaysInGregorianMonth(gMonth, gYear)) {
                diffDays -= getDaysInGregorianMonth(gMonth, gYear);
                gMonth++;
            }
            else {
                gDay += diffDays;
                diffDays = 0;
            }
        }
        else {
            if (diffDays >= (isGregorianLeapYear(gYear - 1) ? 366 : 365)) {
                diffDays -= isGregorianLeapYear(gYear - 1) ? 366 : 365;
                gYear--;
            }
            else {
                if (gMonth > 1) {
                    gMonth--;
                }
                else {
                    gMonth = 12;
                    gYear--;
                }
                if (diffDays >= getDaysInGregorianMonth(gMonth, gYear)) {
                    diffDays -= getDaysInGregorianMonth(gMonth, gYear);
                }
                else {
                    gDay = getDaysInGregorianMonth(gMonth, gYear) - diffDays + 1;
                    diffDays = 0;
                }
            }
        }
    }
    return new Date(gYear, gMonth - 1, gDay);
}
export function hebrewNumerals(numerals) {
    if (!numerals) {
        return '';
    }
    const hArray0_9 = ['', '\u05d0', '\u05d1', '\u05d2', '\u05d3', '\u05d4', '\u05d5', '\u05d6', '\u05d7', '\u05d8'];
    const hArray10_19 = [
        '\u05d9', '\u05d9\u05d0', '\u05d9\u05d1', '\u05d9\u05d2', '\u05d9\u05d3', '\u05d8\u05d5', '\u05d8\u05d6',
        '\u05d9\u05d6', '\u05d9\u05d7', '\u05d9\u05d8'
    ];
    const hArray20_90 = ['', '', '\u05db', '\u05dc', '\u05de', '\u05e0', '\u05e1', '\u05e2', '\u05e4', '\u05e6'];
    const hArray100_900 = [
        '', '\u05e7', '\u05e8', '\u05e9', '\u05ea', '\u05ea\u05e7', '\u05ea\u05e8', '\u05ea\u05e9', '\u05ea\u05ea',
        '\u05ea\u05ea\u05e7'
    ];
    const hArray1000_9000 = [
        '', '\u05d0', '\u05d1', '\u05d1\u05d0', '\u05d1\u05d1', '\u05d4', '\u05d4\u05d0', '\u05d4\u05d1',
        '\u05d4\u05d1\u05d0', '\u05d4\u05d1\u05d1'
    ];
    const geresh = '\u05f3', gershaim = '\u05f4';
    let mem = 0;
    let result = [];
    let step = 0;
    while (numerals > 0) {
        let m = numerals % 10;
        if (step === 0) {
            mem = m;
        }
        else if (step === 1) {
            if (m !== 1) {
                result.unshift(hArray20_90[m], hArray0_9[mem]);
            }
            else {
                result.unshift(hArray10_19[mem]);
            }
        }
        else if (step === 2) {
            result.unshift(hArray100_900[m]);
        }
        else {
            if (m !== 5) {
                result.unshift(hArray1000_9000[m], geresh, ' ');
            }
            break;
        }
        numerals = Math.floor(numerals / 10);
        if (step === 0 && numerals === 0) {
            result.unshift(hArray0_9[m]);
        }
        step++;
    }
    result = result.join('').split('');
    if (result.length === 1) {
        result.push(geresh);
    }
    else if (result.length > 1) {
        result.splice(result.length - 1, 0, gershaim);
    }
    return result.join('');
}
//# sourceMappingURL=data:application/json;base64,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