/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { CompileMetadataResolver, CompilerConfig, CompileReflector, DirectiveNormalizer, DirectiveResolver, DomElementSchemaRegistry, ElementSchemaRegistry, HtmlParser, I18NHtmlParser, JitCompiler, JitEvaluator, JitSummaryResolver, Lexer, NgModuleCompiler, NgModuleResolver, Parser, PipeResolver, ProviderMeta, ResourceLoader, StaticSymbolCache, StyleCompiler, SummaryResolver, TemplateParser, UrlResolver, ViewCompiler } from '@angular/compiler';
import { Compiler, Inject, InjectionToken, Injector, isDevMode, MissingTranslationStrategy, Optional, PACKAGE_ROOT_URL, TRANSLATIONS, TRANSLATIONS_FORMAT, ViewEncapsulation, ɵConsole as Console } from '@angular/core';
import { JitReflector } from './compiler_reflector';
export const ERROR_COLLECTOR_TOKEN = new InjectionToken('ErrorCollector');
/**
 * A default provider for {@link PACKAGE_ROOT_URL} that maps to '/'.
 */
export const DEFAULT_PACKAGE_URL_PROVIDER = {
    provide: PACKAGE_ROOT_URL,
    useValue: '/'
};
const _NO_RESOURCE_LOADER = {
    get(url) {
        throw new Error(`No ResourceLoader implementation has been provided. Can't read the url "${url}"`);
    }
};
const baseHtmlParser = new InjectionToken('HtmlParser');
export class CompilerImpl {
    constructor(injector, _metadataResolver, templateParser, styleCompiler, viewCompiler, ngModuleCompiler, summaryResolver, compileReflector, jitEvaluator, compilerConfig, console) {
        this._metadataResolver = _metadataResolver;
        this._delegate = new JitCompiler(_metadataResolver, templateParser, styleCompiler, viewCompiler, ngModuleCompiler, summaryResolver, compileReflector, jitEvaluator, compilerConfig, console, this.getExtraNgModuleProviders.bind(this));
        this.injector = injector;
    }
    getExtraNgModuleProviders() {
        return [this._metadataResolver.getProviderMetadata(new ProviderMeta(Compiler, { useValue: this }))];
    }
    compileModuleSync(moduleType) {
        return this._delegate.compileModuleSync(moduleType);
    }
    compileModuleAsync(moduleType) {
        return this._delegate.compileModuleAsync(moduleType);
    }
    compileModuleAndAllComponentsSync(moduleType) {
        const result = this._delegate.compileModuleAndAllComponentsSync(moduleType);
        return {
            ngModuleFactory: result.ngModuleFactory,
            componentFactories: result.componentFactories,
        };
    }
    compileModuleAndAllComponentsAsync(moduleType) {
        return this._delegate.compileModuleAndAllComponentsAsync(moduleType)
            .then((result) => ({
            ngModuleFactory: result.ngModuleFactory,
            componentFactories: result.componentFactories,
        }));
    }
    loadAotSummaries(summaries) {
        this._delegate.loadAotSummaries(summaries);
    }
    hasAotSummary(ref) {
        return this._delegate.hasAotSummary(ref);
    }
    getComponentFactory(component) {
        return this._delegate.getComponentFactory(component);
    }
    clearCache() {
        this._delegate.clearCache();
    }
    clearCacheFor(type) {
        this._delegate.clearCacheFor(type);
    }
    getModuleId(moduleType) {
        const meta = this._metadataResolver.getNgModuleMetadata(moduleType);
        return meta && meta.id || undefined;
    }
}
const ɵ0 = new JitReflector(), ɵ1 = _NO_RESOURCE_LOADER, ɵ2 = (parser, translations, format, config, console) => {
    translations = translations || '';
    const missingTranslation = translations ? config.missingTranslation : MissingTranslationStrategy.Ignore;
    return new I18NHtmlParser(parser, translations, format, missingTranslation, console);
}, ɵ3 = new CompilerConfig();
/**
 * A set of providers that provide `JitCompiler` and its dependencies to use for
 * template compilation.
 */
const COMPILER_PROVIDERS__PRE_R3__ = [
    { provide: CompileReflector, useValue: ɵ0 },
    { provide: ResourceLoader, useValue: ɵ1 },
    { provide: JitSummaryResolver, deps: [] },
    { provide: SummaryResolver, useExisting: JitSummaryResolver },
    { provide: Console, deps: [] },
    { provide: Lexer, deps: [] },
    { provide: Parser, deps: [Lexer] },
    {
        provide: baseHtmlParser,
        useClass: HtmlParser,
        deps: [],
    },
    {
        provide: I18NHtmlParser,
        useFactory: ɵ2,
        deps: [
            baseHtmlParser,
            [new Optional(), new Inject(TRANSLATIONS)],
            [new Optional(), new Inject(TRANSLATIONS_FORMAT)],
            [CompilerConfig],
            [Console],
        ]
    },
    {
        provide: HtmlParser,
        useExisting: I18NHtmlParser,
    },
    {
        provide: TemplateParser,
        deps: [CompilerConfig, CompileReflector, Parser, ElementSchemaRegistry, I18NHtmlParser, Console]
    },
    { provide: JitEvaluator, useClass: JitEvaluator, deps: [] },
    { provide: DirectiveNormalizer, deps: [ResourceLoader, UrlResolver, HtmlParser, CompilerConfig] },
    {
        provide: CompileMetadataResolver,
        deps: [
            CompilerConfig, HtmlParser, NgModuleResolver, DirectiveResolver, PipeResolver,
            SummaryResolver, ElementSchemaRegistry, DirectiveNormalizer, Console,
            [Optional, StaticSymbolCache], CompileReflector, [Optional, ERROR_COLLECTOR_TOKEN]
        ]
    },
    DEFAULT_PACKAGE_URL_PROVIDER,
    { provide: StyleCompiler, deps: [UrlResolver] },
    { provide: ViewCompiler, deps: [CompileReflector] },
    { provide: NgModuleCompiler, deps: [CompileReflector] },
    { provide: CompilerConfig, useValue: ɵ3 },
    {
        provide: Compiler,
        useClass: CompilerImpl,
        deps: [
            Injector, CompileMetadataResolver, TemplateParser, StyleCompiler, ViewCompiler,
            NgModuleCompiler, SummaryResolver, CompileReflector, JitEvaluator, CompilerConfig, Console
        ]
    },
    { provide: DomElementSchemaRegistry, deps: [] },
    { provide: ElementSchemaRegistry, useExisting: DomElementSchemaRegistry },
    { provide: UrlResolver, deps: [PACKAGE_ROOT_URL] },
    { provide: DirectiveResolver, deps: [CompileReflector] },
    { provide: PipeResolver, deps: [CompileReflector] },
    { provide: NgModuleResolver, deps: [CompileReflector] },
];
export const COMPILER_PROVIDERS__POST_R3__ = [{ provide: Compiler, useFactory: () => new Compiler() }];
export const COMPILER_PROVIDERS = COMPILER_PROVIDERS__PRE_R3__;
/**
 * @publicApi
 */
export class JitCompilerFactory {
    /* @internal */
    constructor(defaultOptions) {
        const compilerOptions = {
            useJit: true,
            defaultEncapsulation: ViewEncapsulation.Emulated,
            missingTranslation: MissingTranslationStrategy.Warning,
        };
        this._defaultOptions = [compilerOptions, ...defaultOptions];
    }
    createCompiler(options = []) {
        const opts = _mergeOptions(this._defaultOptions.concat(options));
        const injector = Injector.create([
            COMPILER_PROVIDERS, {
                provide: CompilerConfig,
                useFactory: () => {
                    return new CompilerConfig({
                        // let explicit values from the compiler options overwrite options
                        // from the app providers
                        useJit: opts.useJit,
                        jitDevMode: isDevMode(),
                        // let explicit values from the compiler options overwrite options
                        // from the app providers
                        defaultEncapsulation: opts.defaultEncapsulation,
                        missingTranslation: opts.missingTranslation,
                        preserveWhitespaces: opts.preserveWhitespaces,
                    });
                },
                deps: []
            },
            opts.providers
        ]);
        return injector.get(Compiler);
    }
}
function _mergeOptions(optionsArr) {
    return {
        useJit: _lastDefined(optionsArr.map(options => options.useJit)),
        defaultEncapsulation: _lastDefined(optionsArr.map(options => options.defaultEncapsulation)),
        providers: _mergeArrays(optionsArr.map(options => options.providers)),
        missingTranslation: _lastDefined(optionsArr.map(options => options.missingTranslation)),
        preserveWhitespaces: _lastDefined(optionsArr.map(options => options.preserveWhitespaces)),
    };
}
function _lastDefined(args) {
    for (let i = args.length - 1; i >= 0; i--) {
        if (args[i] !== undefined) {
            return args[i];
        }
    }
    return undefined;
}
function _mergeArrays(parts) {
    const result = [];
    parts.forEach((part) => part && result.push(...part));
    return result;
}
export { ɵ0, ɵ1, ɵ2, ɵ3 };
//# sourceMappingURL=data:application/json;base64,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