import { parseTranslation, translate as _translate } from './utils';
/**
 * Load translations for `$localize`.
 *
 * The given `translations` are processed and added to a lookup based on their `MessageId`.
 * A new translation will overwrite a previous translation if it has the same `MessageId`.
 *
 * * If a message is generated by the Angular compiler from an `i18n` marker in a template, the
 *   `MessageId` is passed through to the `$localize` call as a custom `MessageId`. The `MessageId`
 *   will match what is extracted into translation files.
 *
 * * If the translation is from a call to `$localize` in application code, and no custom `MessageId`
 *   is provided, then the `MessageId` can be generated by passing the tagged string message-parts
 *   to the `parseMessage()` function (not currently public API).
 *
 * @publicApi
 *
 */
export function loadTranslations(translations) {
    // Ensure the translate function exists
    if (!$localize.translate) {
        $localize.translate = translate;
    }
    if (!$localize.TRANSLATIONS) {
        $localize.TRANSLATIONS = {};
    }
    Object.keys(translations).forEach(key => {
        $localize.TRANSLATIONS[key] = parseTranslation(translations[key]);
    });
}
/**
 * Remove all translations for `$localize`.
 *
 * @publicApi
 */
export function clearTranslations() {
    $localize.translate = undefined;
    $localize.TRANSLATIONS = {};
}
/**
 * Translate the text of the given message, using the loaded translations.
 *
 * This function may reorder (or remove) substitutions as indicated in the matching translation.
 */
export function translate(messageParts, substitutions) {
    try {
        return _translate($localize.TRANSLATIONS, messageParts, substitutions);
    }
    catch (e) {
        console.warn(e.message);
        return [messageParts, substitutions];
    }
}
//# sourceMappingURL=data:application/json;base64,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