/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { addToArray, removeFromArray } from '../util/array_utils';
import { Services } from './types';
import { declaredViewContainer, renderNode, visitRootRenderNodes } from './util';
export function attachEmbeddedView(parentView, elementData, viewIndex, view) {
    let embeddedViews = elementData.viewContainer._embeddedViews;
    if (viewIndex === null || viewIndex === undefined) {
        viewIndex = embeddedViews.length;
    }
    view.viewContainerParent = parentView;
    addToArray(embeddedViews, viewIndex, view);
    attachProjectedView(elementData, view);
    Services.dirtyParentQueries(view);
    const prevView = viewIndex > 0 ? embeddedViews[viewIndex - 1] : null;
    renderAttachEmbeddedView(elementData, prevView, view);
}
function attachProjectedView(vcElementData, view) {
    const dvcElementData = declaredViewContainer(view);
    if (!dvcElementData || dvcElementData === vcElementData ||
        view.state & 16 /* IsProjectedView */) {
        return;
    }
    // Note: For performance reasons, we
    // - add a view to template._projectedViews only 1x throughout its lifetime,
    //   and remove it not until the view is destroyed.
    //   (hard, as when a parent view is attached/detached we would need to attach/detach all
    //    nested projected views as well, even across component boundaries).
    // - don't track the insertion order of views in the projected views array
    //   (hard, as when the views of the same template are inserted different view containers)
    view.state |= 16 /* IsProjectedView */;
    let projectedViews = dvcElementData.template._projectedViews;
    if (!projectedViews) {
        projectedViews = dvcElementData.template._projectedViews = [];
    }
    projectedViews.push(view);
    // Note: we are changing the NodeDef here as we cannot calculate
    // the fact whether a template is used for projection during compilation.
    markNodeAsProjectedTemplate(view.parent.def, view.parentNodeDef);
}
function markNodeAsProjectedTemplate(viewDef, nodeDef) {
    if (nodeDef.flags & 4 /* ProjectedTemplate */) {
        return;
    }
    viewDef.nodeFlags |= 4 /* ProjectedTemplate */;
    nodeDef.flags |= 4 /* ProjectedTemplate */;
    let parentNodeDef = nodeDef.parent;
    while (parentNodeDef) {
        parentNodeDef.childFlags |= 4 /* ProjectedTemplate */;
        parentNodeDef = parentNodeDef.parent;
    }
}
export function detachEmbeddedView(elementData, viewIndex) {
    const embeddedViews = elementData.viewContainer._embeddedViews;
    if (viewIndex == null || viewIndex >= embeddedViews.length) {
        viewIndex = embeddedViews.length - 1;
    }
    if (viewIndex < 0) {
        return null;
    }
    const view = embeddedViews[viewIndex];
    view.viewContainerParent = null;
    removeFromArray(embeddedViews, viewIndex);
    // See attachProjectedView for why we don't update projectedViews here.
    Services.dirtyParentQueries(view);
    renderDetachView(view);
    return view;
}
export function detachProjectedView(view) {
    if (!(view.state & 16 /* IsProjectedView */)) {
        return;
    }
    const dvcElementData = declaredViewContainer(view);
    if (dvcElementData) {
        const projectedViews = dvcElementData.template._projectedViews;
        if (projectedViews) {
            removeFromArray(projectedViews, projectedViews.indexOf(view));
            Services.dirtyParentQueries(view);
        }
    }
}
export function moveEmbeddedView(elementData, oldViewIndex, newViewIndex) {
    const embeddedViews = elementData.viewContainer._embeddedViews;
    const view = embeddedViews[oldViewIndex];
    removeFromArray(embeddedViews, oldViewIndex);
    if (newViewIndex == null) {
        newViewIndex = embeddedViews.length;
    }
    addToArray(embeddedViews, newViewIndex, view);
    // Note: Don't need to change projectedViews as the order in there
    // as always invalid...
    Services.dirtyParentQueries(view);
    renderDetachView(view);
    const prevView = newViewIndex > 0 ? embeddedViews[newViewIndex - 1] : null;
    renderAttachEmbeddedView(elementData, prevView, view);
    return view;
}
function renderAttachEmbeddedView(elementData, prevView, view) {
    const prevRenderNode = prevView ? renderNode(prevView, prevView.def.lastRenderRootNode) : elementData.renderElement;
    const parentNode = view.renderer.parentNode(prevRenderNode);
    const nextSibling = view.renderer.nextSibling(prevRenderNode);
    // Note: We can't check if `nextSibling` is present, as on WebWorkers it will always be!
    // However, browsers automatically do `appendChild` when there is no `nextSibling`.
    visitRootRenderNodes(view, 2 /* InsertBefore */, parentNode, nextSibling, undefined);
}
export function renderDetachView(view) {
    visitRootRenderNodes(view, 3 /* RemoveChild */, null, null, undefined);
}
//# sourceMappingURL=data:application/json;base64,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