/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { SimpleChange } from '../../interface/simple_change';
import { EMPTY_OBJ } from '../empty';
/**
 * The NgOnChangesFeature decorates a component with support for the ngOnChanges
 * lifecycle hook, so it should be included in any component that implements
 * that hook.
 *
 * If the component or directive uses inheritance, the NgOnChangesFeature MUST
 * be included as a feature AFTER {@link InheritDefinitionFeature}, otherwise
 * inherited properties will not be propagated to the ngOnChanges lifecycle
 * hook.
 *
 * Example usage:
 *
 * ```
 * static ɵcmp = defineComponent({
 *   ...
 *   inputs: {name: 'publicName'},
 *   features: [NgOnChangesFeature]
 * });
 * ```
 *
 * @codeGenApi
 */
export function ɵɵNgOnChangesFeature() {
    return NgOnChangesFeatureImpl;
}
export function NgOnChangesFeatureImpl(definition) {
    if (definition.type.prototype.ngOnChanges) {
        definition.setInput = ngOnChangesSetInput;
    }
    return rememberChangeHistoryAndInvokeOnChangesHook;
}
// This option ensures that the ngOnChanges lifecycle hook will be inherited
// from superclasses (in InheritDefinitionFeature).
/** @nocollapse */
// tslint:disable-next-line:no-toplevel-property-access
ɵɵNgOnChangesFeature.ngInherit = true;
/**
 * This is a synthetic lifecycle hook which gets inserted into `TView.preOrderHooks` to simulate
 * `ngOnChanges`.
 *
 * The hook reads the `NgSimpleChangesStore` data from the component instance and if changes are
 * found it invokes `ngOnChanges` on the component instance.
 *
 * @param this Component instance. Because this function gets inserted into `TView.preOrderHooks`,
 *     it is guaranteed to be called with component instance.
 */
function rememberChangeHistoryAndInvokeOnChangesHook() {
    const simpleChangesStore = getSimpleChangesStore(this);
    const current = simpleChangesStore === null || simpleChangesStore === void 0 ? void 0 : simpleChangesStore.current;
    if (current) {
        const previous = simpleChangesStore.previous;
        if (previous === EMPTY_OBJ) {
            simpleChangesStore.previous = current;
        }
        else {
            // New changes are copied to the previous store, so that we don't lose history for inputs
            // which were not changed this time
            for (let key in current) {
                previous[key] = current[key];
            }
        }
        simpleChangesStore.current = null;
        this.ngOnChanges(current);
    }
}
function ngOnChangesSetInput(instance, value, publicName, privateName) {
    const simpleChangesStore = getSimpleChangesStore(instance) ||
        setSimpleChangesStore(instance, { previous: EMPTY_OBJ, current: null });
    const current = simpleChangesStore.current || (simpleChangesStore.current = {});
    const previous = simpleChangesStore.previous;
    const declaredName = this.declaredInputs[publicName];
    const previousChange = previous[declaredName];
    current[declaredName] = new SimpleChange(previousChange && previousChange.currentValue, value, previous === EMPTY_OBJ);
    instance[privateName] = value;
}
const SIMPLE_CHANGES_STORE = '__ngSimpleChanges__';
function getSimpleChangesStore(instance) {
    return instance[SIMPLE_CHANGES_STORE] || null;
}
function setSimpleChangesStore(instance, store) {
    return instance[SIMPLE_CHANGES_STORE] = store;
}
//# sourceMappingURL=data:application/json;base64,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