import { stringify } from '../util/stringify';
import { TVIEW } from './interfaces/view';
import { INTERPOLATION_DELIMITER, stringifyForError } from './util/misc_utils';
/** Called when directives inject each other (creating a circular dependency) */
export function throwCyclicDependencyError(token, path) {
    const depPath = path ? `. Dependency path: ${path.join(' > ')} > ${token}` : '';
    throw new Error(`Circular dependency in DI detected for ${token}${depPath}`);
}
/** Called when there are multiple component selectors that match a given node */
export function throwMultipleComponentError(tNode) {
    throw new Error(`Multiple components match node with tagname ${tNode.tagName}`);
}
export function throwMixedMultiProviderError() {
    throw new Error(`Cannot mix multi providers and regular providers`);
}
export function throwInvalidProviderError(ngModuleType, providers, provider) {
    let ngModuleDetail = '';
    if (ngModuleType && providers) {
        const providerDetail = providers.map(v => v == provider ? '?' + provider + '?' : '...');
        ngModuleDetail =
            ` - only instances of Provider and Type are allowed, got: [${providerDetail.join(', ')}]`;
    }
    throw new Error(`Invalid provider for the NgModule '${stringify(ngModuleType)}'` + ngModuleDetail);
}
/** Throws an ExpressionChangedAfterChecked error if checkNoChanges mode is on. */
export function throwErrorIfNoChangesMode(creationMode, oldValue, currValue, propName) {
    const field = propName ? ` for '${propName}'` : '';
    let msg = `ExpressionChangedAfterItHasBeenCheckedError: Expression has changed after it was checked. Previous value${field}: '${oldValue}'. Current value: '${currValue}'.`;
    if (creationMode) {
        msg +=
            ` It seems like the view has been created after its parent and its children have been dirty checked.` +
                ` Has it been created in a change detection hook?`;
    }
    // TODO: include debug context, see `viewDebugError` function in
    // `packages/core/src/view/errors.ts` for reference.
    throw new Error(msg);
}
function constructDetailsForInterpolation(lView, rootIndex, expressionIndex, meta, changedValue) {
    const [propName, prefix, ...chunks] = meta.split(INTERPOLATION_DELIMITER);
    let oldValue = prefix, newValue = prefix;
    for (let i = 0; i < chunks.length; i++) {
        const slotIdx = rootIndex + i;
        oldValue += `${lView[slotIdx]}${chunks[i]}`;
        newValue += `${slotIdx === expressionIndex ? changedValue : lView[slotIdx]}${chunks[i]}`;
    }
    return { propName, oldValue, newValue };
}
/**
 * Constructs an object that contains details for the ExpressionChangedAfterItHasBeenCheckedError:
 * - property name (for property bindings or interpolations)
 * - old and new values, enriched using information from metadata
 *
 * More information on the metadata storage format can be found in `storePropertyBindingMetadata`
 * function description.
 */
export function getExpressionChangedErrorDetails(lView, bindingIndex, oldValue, newValue) {
    const tData = lView[TVIEW].data;
    const metadata = tData[bindingIndex];
    if (typeof metadata === 'string') {
        // metadata for property interpolation
        if (metadata.indexOf(INTERPOLATION_DELIMITER) > -1) {
            return constructDetailsForInterpolation(lView, bindingIndex, bindingIndex, metadata, newValue);
        }
        // metadata for property binding
        return { propName: metadata, oldValue, newValue };
    }
    // metadata is not available for this expression, check if this expression is a part of the
    // property interpolation by going from the current binding index left and look for a string that
    // contains INTERPOLATION_DELIMITER, the layout in tView.data for this case will look like this:
    // [..., 'id�Prefix � and � suffix', null, null, null, ...]
    if (metadata === null) {
        let idx = bindingIndex - 1;
        while (typeof tData[idx] !== 'string' && tData[idx + 1] === null) {
            idx--;
        }
        const meta = tData[idx];
        if (typeof meta === 'string') {
            const matches = meta.match(new RegExp(INTERPOLATION_DELIMITER, 'g'));
            // first interpolation delimiter separates property name from interpolation parts (in case of
            // property interpolations), so we subtract one from total number of found delimiters
            if (matches && (matches.length - 1) > bindingIndex - idx) {
                return constructDetailsForInterpolation(lView, idx, bindingIndex, meta, newValue);
            }
        }
    }
    return { propName: undefined, oldValue, newValue };
}
/** Throws an error when a token is not found in DI. */
export function throwProviderNotFoundError(token, injectorName) {
    const injectorDetails = injectorName ? ` in ${injectorName}` : '';
    throw new Error(`No provider for ${stringifyForError(token)} found${injectorDetails}`);
}
//# sourceMappingURL=data:application/json;base64,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