/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { EMPTY_ARRAY, EMPTY_OBJ } from './empty';
import { NG_COMP_DEF, NG_DIR_DEF, NG_FACTORY_DEF, NG_LOC_ID_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
let _renderCompCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const type = componentDefinition.type;
        const typePrototype = type.prototype;
        const declaredInputs = {};
        const def = {
            type: type,
            providersResolver: null,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            factory: null,
            template: componentDefinition.template || null,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            hostBindings: componentDefinition.hostBindings || null,
            hostVars: componentDefinition.hostVars || 0,
            hostAttrs: componentDefinition.hostAttrs || null,
            contentQueries: componentDefinition.contentQueries || null,
            declaredInputs: declaredInputs,
            inputs: null,
            outputs: null,
            exportAs: componentDefinition.exportAs || null,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            selectors: componentDefinition.selectors || EMPTY_ARRAY,
            viewQuery: componentDefinition.viewQuery || null,
            features: componentDefinition.features || null,
            data: componentDefinition.data || {},
            // TODO(misko): convert ViewEncapsulation into const enum so that it can be used
            // directly in the next line. Also `None` should be 0 not 2.
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            id: 'c',
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            setInput: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
        };
        const directiveTypes = componentDefinition.directives;
        const feature = componentDefinition.features;
        const pipeTypes = componentDefinition.pipes;
        def.id += _renderCompCount++;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach((fn) => fn(def));
        def.directiveDefs = directiveTypes ?
            () => (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef) :
            null;
        def.pipeDefs = pipeTypes ?
            () => (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef) :
            null;
        return def;
    });
}
/**
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = () => directives.map(extractDirectiveDef);
    def.pipeDefs = () => pipes.map(extractPipeDef);
}
export function extractDirectiveDef(type) {
    const def = getComponentDef(type) || getDirectiveDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is neither 'ComponentType' or 'DirectiveType'.`);
    }
    return def;
}
export function extractPipeDef(type) {
    const def = getPipeDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is not a 'PipeType'.`);
    }
    return def;
}
export const autoRegisterModuleById = {};
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    const res = {
        type: def.type,
        bootstrap: def.bootstrap || EMPTY_ARRAY,
        declarations: def.declarations || EMPTY_ARRAY,
        imports: def.imports || EMPTY_ARRAY,
        exports: def.exports || EMPTY_ARRAY,
        transitiveCompileScopes: null,
        schemas: def.schemas || null,
        id: def.id || null,
    };
    if (def.id != null) {
        noSideEffects(() => {
            autoRegisterModuleById[def.id] = def.type;
        });
    }
    return res;
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export const ɵɵdefineDirective = ɵɵdefineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition form the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
export function getFactoryDef(type, throwNotFound) {
    const hasFactoryDef = type.hasOwnProperty(NG_FACTORY_DEF);
    if (!hasFactoryDef && throwNotFound === true && ngDevMode) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵfac' property.`);
    }
    return hasFactoryDef ? type[NG_FACTORY_DEF] : null;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
export function getNgLocaleIdDef(type) {
    return type[NG_LOC_ID_DEF] || null;
}
//# sourceMappingURL=data:application/json;base64,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