/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent as render3CompileComponent, compileDirective as render3CompileDirective } from '../render3/jit/directive';
import { compilePipe as render3CompilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
import { noop } from '../util/noop';
const ɵ0 = (dir = {}) => dir, ɵ1 = (type, meta) => SWITCH_COMPILE_DIRECTIVE(type, meta);
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', ɵ0, undefined, undefined, ɵ1);
const ɵ2 = (c = {}) => (Object.assign({ changeDetection: ChangeDetectionStrategy.Default }, c)), ɵ3 = (type, meta) => SWITCH_COMPILE_COMPONENT(type, meta);
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', ɵ2, Directive, undefined, ɵ3);
const ɵ4 = (p) => (Object.assign({ pure: true }, p)), ɵ5 = (type, meta) => SWITCH_COMPILE_PIPE(type, meta);
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', ɵ4, undefined, undefined, ɵ5);
const ɵ6 = (bindingPropertyName) => ({ bindingPropertyName });
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', ɵ6);
const ɵ7 = (bindingPropertyName) => ({ bindingPropertyName });
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', ɵ7);
const ɵ8 = (hostPropertyName) => ({ hostPropertyName });
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', ɵ8);
const ɵ9 = (eventName, args) => ({ eventName, args });
/**
 * Decorator that binds a DOM event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 *
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```ts
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *  }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 *
 * ```
 *
 * The following example registers another DOM event handler that listens for key-press events.
 * ``` ts
 * import { HostListener, Component } from "@angular/core";
 *
 * @Component({
 *   selector: 'app',
 *   template: `<h1>Hello, you have pressed keys {{counter}} number of times!</h1> Press any key to
 * increment the counter.
 *   <button (click)="resetCounter()">Reset Counter</button>`
 * })
 * class AppComponent {
 *   counter = 0;
 *   @HostListener('window:keydown', ['$event'])
 *   handleKeyDown(event: KeyboardEvent) {
 *     this.counter++;
 *   }
 *   resetCounter() {
 *     this.counter = 0;
 *   }
 * }
 * ```
 *
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', ɵ9);
export const SWITCH_COMPILE_COMPONENT__POST_R3__ = render3CompileComponent;
export const SWITCH_COMPILE_DIRECTIVE__POST_R3__ = render3CompileDirective;
export const SWITCH_COMPILE_PIPE__POST_R3__ = render3CompilePipe;
const SWITCH_COMPILE_COMPONENT__PRE_R3__ = noop;
const SWITCH_COMPILE_DIRECTIVE__PRE_R3__ = noop;
const SWITCH_COMPILE_PIPE__PRE_R3__ = noop;
const SWITCH_COMPILE_COMPONENT = SWITCH_COMPILE_COMPONENT__PRE_R3__;
const SWITCH_COMPILE_DIRECTIVE = SWITCH_COMPILE_DIRECTIVE__PRE_R3__;
const SWITCH_COMPILE_PIPE = SWITCH_COMPILE_PIPE__PRE_R3__;
export { ɵ0, ɵ1, ɵ2, ɵ3, ɵ4, ɵ5, ɵ6, ɵ7, ɵ8, ɵ9 };
//# sourceMappingURL=data:application/json;base64,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