/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getClosureSafeProperty } from '../../util/property';
/**
 * Construct an `InjectableDef` which defines how a token will be constructed by the DI system, and
 * in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call `inject` to access the `Injector` and request injection of dependencies.
 *
 * @codeGenApi
 * @publicApi This instruction has been emitted by ViewEngine for some time and is deployed to npm.
 */
export function ɵɵdefineInjectable(opts) {
    return {
        token: opts.token,
        providedIn: opts.providedIn || null,
        factory: opts.factory,
        value: undefined,
    };
}
/**
 * @deprecated in v8, delete after v10. This API should be used only be generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export const defineInjectable = ɵɵdefineInjectable;
/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `factory`: an `InjectorType` is an instantiable type, so a zero argument `factory` function to
 *   create the type must be provided. If that factory function needs to inject arguments, it can
 *   use the `inject` function.
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @codeGenApi
 */
export function ɵɵdefineInjector(options) {
    return {
        factory: options.factory,
        providers: options.providers || [],
        imports: options.imports || [],
    };
}
/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export function getInjectableDef(type) {
    return getOwnDefinition(type, type[NG_PROV_DEF]) ||
        getOwnDefinition(type, type[NG_INJECTABLE_DEF]);
}
/**
 * Return `def` only if it is defined directly on `type` and is not inherited from a base
 * class of `type`.
 *
 * The function `Object.hasOwnProperty` is not sufficient to distinguish this case because in older
 * browsers (e.g. IE10) static property inheritance is implemented by copying the properties.
 *
 * Instead, the definition's `token` is compared to the `type`, and if they don't match then the
 * property was not defined directly on the type itself, and was likely inherited. The definition
 * is only returned if the `type` matches the `def.token`.
 */
function getOwnDefinition(type, def) {
    return def && def.token === type ? def : null;
}
/**
 * Read the injectable def (`ɵprov`) for `type` or read the `ɵprov` from one of its ancestors.
 *
 * @param type A type which may have `ɵprov`, via inheritance.
 *
 * @deprecated Will be removed in a future version of Angular, where an error will occur in the
 *     scenario if we find the `ɵprov` on an ancestor only.
 */
export function getInheritedInjectableDef(type) {
    // See `jit/injectable.ts#compileInjectable` for context on NG_PROV_DEF_FALLBACK.
    const def = type &&
        (type[NG_PROV_DEF] || type[NG_INJECTABLE_DEF] ||
            (type[NG_PROV_DEF_FALLBACK] && type[NG_PROV_DEF_FALLBACK]()));
    if (def) {
        const typeName = getTypeName(type);
        // TODO(FW-1307): Re-add ngDevMode when closure can handle it
        // ngDevMode &&
        console.warn(`DEPRECATED: DI is instantiating a token "${typeName}" that inherits its @Injectable decorator but does not provide one itself.\n` +
            `This will become an error in a future version of Angular. Please add @Injectable() to the "${typeName}" class.`);
        return def;
    }
    else {
        return null;
    }
}
/** Gets the name of a type, accounting for some cross-browser differences. */
function getTypeName(type) {
    // `Function.prototype.name` behaves differently between IE and other browsers. In most browsers
    // it'll always return the name of the function itself, no matter how many other functions it
    // inherits from. On IE the function doesn't have its own `name` property, but it takes it from
    // the lowest level in the prototype chain. E.g. if we have `class Foo extends Parent` most
    // browsers will evaluate `Foo.name` to `Foo` while IE will return `Parent`. We work around
    // the issue by converting the function to a string and parsing its name out that way via a regex.
    if (type.hasOwnProperty('name')) {
        return type.name;
    }
    const match = ('' + type).match(/^function\s*([^\s(]+)/);
    return match === null ? '' : match[1];
}
/**
 * Read the injector def type in a way which is immune to accidentally reading inherited value.
 *
 * @param type type which may have an injector def (`ɵinj`)
 */
export function getInjectorDef(type) {
    return type && (type.hasOwnProperty(NG_INJ_DEF) || type.hasOwnProperty(NG_INJECTOR_DEF)) ?
        type[NG_INJ_DEF] :
        null;
}
export const NG_PROV_DEF = getClosureSafeProperty({ ɵprov: getClosureSafeProperty });
export const NG_INJ_DEF = getClosureSafeProperty({ ɵinj: getClosureSafeProperty });
// On IE10 properties defined via `defineProperty` won't be inherited by child classes,
// which will break inheriting the injectable definition from a grandparent through an
// undecorated parent class. We work around it by defining a fallback method which will be
// used to retrieve the definition. This should only be a problem in JIT mode, because in
// AOT TypeScript seems to have a workaround for static properties. When inheriting from an
// undecorated parent is no longer supported in v10, this can safely be removed.
export const NG_PROV_DEF_FALLBACK = getClosureSafeProperty({ ɵprovFallback: getClosureSafeProperty });
// We need to keep these around so we can read off old defs if new defs are unavailable
export const NG_INJECTABLE_DEF = getClosureSafeProperty({ ngInjectableDef: getClosureSafeProperty });
export const NG_INJECTOR_DEF = getClosureSafeProperty({ ngInjectorDef: getClosureSafeProperty });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVmcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL2RpL2ludGVyZmFjZS9kZWZzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQUdILE9BQU8sRUFBQyxzQkFBc0IsRUFBQyxNQUFNLHFCQUFxQixDQUFDO0FBb0gzRDs7Ozs7Ozs7Ozs7Ozs7OztHQWdCRztBQUNILE1BQU0sVUFBVSxrQkFBa0IsQ0FBSSxJQUdyQztJQUNDLE9BQVE7UUFDQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7UUFDakIsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFpQixJQUFJLElBQUk7UUFDMUMsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPO1FBQ3JCLEtBQUssRUFBRSxTQUFTO0tBQ2UsQ0FBQztBQUMzQyxDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILE1BQU0sQ0FBQyxNQUFNLGdCQUFnQixHQUFHLGtCQUFrQixDQUFDO0FBRW5EOzs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBbUJHO0FBQ0gsTUFBTSxVQUFVLGdCQUFnQixDQUFDLE9BQWlFO0lBRWhHLE9BQVE7UUFDQyxPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU87UUFDeEIsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTLElBQUksRUFBRTtRQUNsQyxPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU8sSUFBSSxFQUFFO0tBQ0MsQ0FBQztBQUMzQyxDQUFDO0FBRUQ7Ozs7O0dBS0c7QUFDSCxNQUFNLFVBQVUsZ0JBQWdCLENBQUksSUFBUztJQUMzQyxPQUFPLGdCQUFnQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDNUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUM7QUFDdEQsQ0FBQztBQUVEOzs7Ozs7Ozs7O0dBVUc7QUFDSCxTQUFTLGdCQUFnQixDQUFJLElBQVMsRUFBRSxHQUF1QjtJQUM3RCxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7QUFDaEQsQ0FBQztBQUVEOzs7Ozs7O0dBT0c7QUFDSCxNQUFNLFVBQVUseUJBQXlCLENBQUksSUFBUztJQUNwRCxpRkFBaUY7SUFDakYsTUFBTSxHQUFHLEdBQUcsSUFBSTtRQUNaLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztZQUM1QyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBRW5FLElBQUksR0FBRyxFQUFFO1FBQ1AsTUFBTSxRQUFRLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ25DLDZEQUE2RDtRQUM3RCxlQUFlO1FBQ2YsT0FBTyxDQUFDLElBQUksQ0FDUiw0Q0FDSSxRQUFRLDhFQUE4RTtZQUMxRiw4RkFDSSxRQUFRLFVBQVUsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sR0FBRyxDQUFDO0tBQ1o7U0FBTTtRQUNMLE9BQU8sSUFBSSxDQUFDO0tBQ2I7QUFDSCxDQUFDO0FBRUQsOEVBQThFO0FBQzlFLFNBQVMsV0FBVyxDQUFDLElBQVM7SUFDNUIsZ0dBQWdHO0lBQ2hHLDZGQUE2RjtJQUM3RiwrRkFBK0Y7SUFDL0YsMkZBQTJGO0lBQzNGLDJGQUEyRjtJQUMzRixrR0FBa0c7SUFDbEcsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1FBQy9CLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztLQUNsQjtJQUVELE1BQU0sS0FBSyxHQUFHLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQ3pELE9BQU8sS0FBSyxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDeEMsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxNQUFNLFVBQVUsY0FBYyxDQUFJLElBQVM7SUFDekMsT0FBTyxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JGLElBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBQzNCLElBQUksQ0FBQztBQUNYLENBQUM7QUFFRCxNQUFNLENBQUMsTUFBTSxXQUFXLEdBQUcsc0JBQXNCLENBQUMsRUFBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUMsQ0FBQyxDQUFDO0FBQ25GLE1BQU0sQ0FBQyxNQUFNLFVBQVUsR0FBRyxzQkFBc0IsQ0FBQyxFQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBQyxDQUFDLENBQUM7QUFFakYsdUZBQXVGO0FBQ3ZGLHNGQUFzRjtBQUN0RiwwRkFBMEY7QUFDMUYseUZBQXlGO0FBQ3pGLDJGQUEyRjtBQUMzRixnRkFBZ0Y7QUFDaEYsTUFBTSxDQUFDLE1BQU0sb0JBQW9CLEdBQUcsc0JBQXNCLENBQUMsRUFBQyxhQUFhLEVBQUUsc0JBQXNCLEVBQUMsQ0FBQyxDQUFDO0FBRXBHLHVGQUF1RjtBQUN2RixNQUFNLENBQUMsTUFBTSxpQkFBaUIsR0FBRyxzQkFBc0IsQ0FBQyxFQUFDLGVBQWUsRUFBRSxzQkFBc0IsRUFBQyxDQUFDLENBQUM7QUFDbkcsTUFBTSxDQUFDLE1BQU0sZUFBZSxHQUFHLHNCQUFzQixDQUFDLEVBQUMsYUFBYSxFQUFFLHNCQUFzQixFQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge1R5cGV9IGZyb20gJy4uLy4uL2ludGVyZmFjZS90eXBlJztcbmltcG9ydCB7Z2V0Q2xvc3VyZVNhZmVQcm9wZXJ0eX0gZnJvbSAnLi4vLi4vdXRpbC9wcm9wZXJ0eSc7XG5pbXBvcnQge0NsYXNzUHJvdmlkZXIsIENvbnN0cnVjdG9yUHJvdmlkZXIsIEV4aXN0aW5nUHJvdmlkZXIsIEZhY3RvcnlQcm92aWRlciwgU3RhdGljQ2xhc3NQcm92aWRlciwgVmFsdWVQcm92aWRlcn0gZnJvbSAnLi9wcm92aWRlcic7XG5cblxuXG4vKipcbiAqIEluZm9ybWF0aW9uIGFib3V0IGhvdyBhIHR5cGUgb3IgYEluamVjdGlvblRva2VuYCBpbnRlcmZhY2VzIHdpdGggdGhlIERJIHN5c3RlbS5cbiAqXG4gKiBBdCBhIG1pbmltdW0sIHRoaXMgaW5jbHVkZXMgYSBgZmFjdG9yeWAgd2hpY2ggZGVmaW5lcyBob3cgdG8gY3JlYXRlIHRoZSBnaXZlbiB0eXBlIGBUYCwgcG9zc2libHlcbiAqIHJlcXVlc3RpbmcgaW5qZWN0aW9uIG9mIG90aGVyIHR5cGVzIGlmIG5lY2Vzc2FyeS5cbiAqXG4gKiBPcHRpb25hbGx5LCBhIGBwcm92aWRlZEluYCBwYXJhbWV0ZXIgc3BlY2lmaWVzIHRoYXQgdGhlIGdpdmVuIHR5cGUgYmVsb25ncyB0byBhIHBhcnRpY3VsYXJcbiAqIGBJbmplY3RvckRlZmAsIGBOZ01vZHVsZWAsIG9yIGEgc3BlY2lhbCBzY29wZSAoZS5nLiBgJ3Jvb3QnYCkuIEEgdmFsdWUgb2YgYG51bGxgIGluZGljYXRlc1xuICogdGhhdCB0aGUgaW5qZWN0YWJsZSBkb2VzIG5vdCBiZWxvbmcgdG8gYW55IHNjb3BlLlxuICpcbiAqIEBjb2RlR2VuQXBpXG4gKiBAcHVibGljQXBpIFRoZSBWaWV3RW5naW5lIGNvbXBpbGVyIGVtaXRzIGNvZGUgd2l0aCB0aGlzIHR5cGUgZm9yIGluamVjdGFibGVzLiBUaGlzIGNvZGUgaXNcbiAqICAgZGVwbG95ZWQgdG8gbnBtLCBhbmQgc2hvdWxkIGJlIHRyZWF0ZWQgYXMgcHVibGljIGFwaS5cblxuICovXG5leHBvcnQgaW50ZXJmYWNlIMm1ybVJbmplY3RhYmxlRGVmPFQ+IHtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGF0IHRoZSBnaXZlbiB0eXBlIGJlbG9uZ3MgdG8gYSBwYXJ0aWN1bGFyIGluamVjdG9yOlxuICAgKiAtIGBJbmplY3RvclR5cGVgIHN1Y2ggYXMgYE5nTW9kdWxlYCxcbiAgICogLSBgJ3Jvb3QnYCB0aGUgcm9vdCBpbmplY3RvclxuICAgKiAtIGAnYW55J2AgYWxsIGluamVjdG9ycy5cbiAgICogLSBgbnVsbGAsIGRvZXMgbm90IGJlbG9uZyB0byBhbnkgaW5qZWN0b3IuIE11c3QgYmUgZXhwbGljaXRseSBsaXN0ZWQgaW4gdGhlIGluamVjdG9yXG4gICAqICAgYHByb3ZpZGVyc2AuXG4gICAqL1xuICBwcm92aWRlZEluOiBJbmplY3RvclR5cGU8YW55Pnwncm9vdCd8J3BsYXRmb3JtJ3wnYW55J3xudWxsO1xuXG4gIC8qKlxuICAgKiBUaGUgdG9rZW4gdG8gd2hpY2ggdGhpcyBkZWZpbml0aW9uIGJlbG9uZ3MuXG4gICAqXG4gICAqIE5vdGUgdGhhdCB0aGlzIG1heSBub3QgYmUgdGhlIHNhbWUgYXMgdGhlIHR5cGUgdGhhdCB0aGUgYGZhY3RvcnlgIHdpbGwgY3JlYXRlLlxuICAgKi9cbiAgdG9rZW46IHVua25vd247XG5cbiAgLyoqXG4gICAqIEZhY3RvcnkgbWV0aG9kIHRvIGV4ZWN1dGUgdG8gY3JlYXRlIGFuIGluc3RhbmNlIG9mIHRoZSBpbmplY3RhYmxlLlxuICAgKi9cbiAgZmFjdG9yeTogKHQ/OiBUeXBlPGFueT4pID0+IFQ7XG5cbiAgLyoqXG4gICAqIEluIGEgY2FzZSBvZiBubyBleHBsaWNpdCBpbmplY3RvciwgYSBsb2NhdGlvbiB3aGVyZSB0aGUgaW5zdGFuY2Ugb2YgdGhlIGluamVjdGFibGUgaXMgc3RvcmVkLlxuICAgKi9cbiAgdmFsdWU6IFR8dW5kZWZpbmVkO1xufVxuXG4vKipcbiAqIEluZm9ybWF0aW9uIGFib3V0IHRoZSBwcm92aWRlcnMgdG8gYmUgaW5jbHVkZWQgaW4gYW4gYEluamVjdG9yYCBhcyB3ZWxsIGFzIGhvdyB0aGUgZ2l2ZW4gdHlwZVxuICogd2hpY2ggY2FycmllcyB0aGUgaW5mb3JtYXRpb24gc2hvdWxkIGJlIGNyZWF0ZWQgYnkgdGhlIERJIHN5c3RlbS5cbiAqXG4gKiBBbiBgSW5qZWN0b3JEZWZgIGNhbiBpbXBvcnQgb3RoZXIgdHlwZXMgd2hpY2ggaGF2ZSBgSW5qZWN0b3JEZWZzYCwgZm9ybWluZyBhIGRlZXAgbmVzdGVkXG4gKiBzdHJ1Y3R1cmUgb2YgcHJvdmlkZXJzIHdpdGggYSBkZWZpbmVkIHByaW9yaXR5IChpZGVudGljYWxseSB0byBob3cgYE5nTW9kdWxlYHMgYWxzbyBoYXZlXG4gKiBhbiBpbXBvcnQvZGVwZW5kZW5jeSBzdHJ1Y3R1cmUpLlxuICpcbiAqIE5PVEU6IFRoaXMgaXMgYSBwcml2YXRlIHR5cGUgYW5kIHNob3VsZCBub3QgYmUgZXhwb3J0ZWRcbiAqXG4gKiBAY29kZUdlbkFwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIMm1ybVJbmplY3RvckRlZjxUPiB7XG4gIGZhY3Rvcnk6ICgpID0+IFQ7XG5cbiAgLy8gVE9ETyhhbHhodWIpOiBOYXJyb3cgZG93biB0aGUgdHlwZSBoZXJlIG9uY2UgZGVjb3JhdG9ycyBwcm9wZXJseSBjaGFuZ2UgdGhlIHJldHVybiB0eXBlIG9mIHRoZVxuICAvLyBjbGFzcyB0aGV5IGFyZSBkZWNvcmF0aW5nICh0byBhZGQgdGhlIMm1cHJvdiBwcm9wZXJ0eSBmb3IgZXhhbXBsZSkuXG4gIHByb3ZpZGVyczogKFR5cGU8YW55PnxWYWx1ZVByb3ZpZGVyfEV4aXN0aW5nUHJvdmlkZXJ8RmFjdG9yeVByb3ZpZGVyfENvbnN0cnVjdG9yUHJvdmlkZXJ8XG4gICAgICAgICAgICAgIFN0YXRpY0NsYXNzUHJvdmlkZXJ8Q2xhc3NQcm92aWRlcnxhbnlbXSlbXTtcblxuICBpbXBvcnRzOiAoSW5qZWN0b3JUeXBlPGFueT58SW5qZWN0b3JUeXBlV2l0aFByb3ZpZGVyczxhbnk+KVtdO1xufVxuXG4vKipcbiAqIEEgYFR5cGVgIHdoaWNoIGhhcyBhbiBgSW5qZWN0YWJsZURlZmAgc3RhdGljIGZpZWxkLlxuICpcbiAqIGBJbmplY3RhYmxlRGVmVHlwZWBzIGNvbnRhaW4gdGhlaXIgb3duIERlcGVuZGVuY3kgSW5qZWN0aW9uIG1ldGFkYXRhIGFuZCBhcmUgdXNhYmxlIGluIGFuXG4gKiBgSW5qZWN0b3JEZWZgLWJhc2VkIGBTdGF0aWNJbmplY3Rvci5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW5qZWN0YWJsZVR5cGU8VD4gZXh0ZW5kcyBUeXBlPFQ+IHtcbiAgLyoqXG4gICAqIE9wYXF1ZSB0eXBlIHdob3NlIHN0cnVjdHVyZSBpcyBoaWdobHkgdmVyc2lvbiBkZXBlbmRlbnQuIERvIG5vdCByZWx5IG9uIGFueSBwcm9wZXJ0aWVzLlxuICAgKi9cbiAgybVwcm92OiBuZXZlcjtcbn1cblxuLyoqXG4gKiBBIHR5cGUgd2hpY2ggaGFzIGFuIGBJbmplY3RvckRlZmAgc3RhdGljIGZpZWxkLlxuICpcbiAqIGBJbmplY3RvckRlZlR5cGVzYCBjYW4gYmUgdXNlZCB0byBjb25maWd1cmUgYSBgU3RhdGljSW5qZWN0b3JgLlxuICpcbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbmplY3RvclR5cGU8VD4gZXh0ZW5kcyBUeXBlPFQ+IHtcbiAgLyoqXG4gICAqIE9wYXF1ZSB0eXBlIHdob3NlIHN0cnVjdHVyZSBpcyBoaWdobHkgdmVyc2lvbiBkZXBlbmRlbnQuIERvIG5vdCByZWx5IG9uIGFueSBwcm9wZXJ0aWVzLlxuICAgKi9cbiAgybVpbmo6IG5ldmVyO1xufVxuXG4vKipcbiAqIERlc2NyaWJlcyB0aGUgYEluamVjdG9yRGVmYCBlcXVpdmFsZW50IG9mIGEgYE1vZHVsZVdpdGhQcm92aWRlcnNgLCBhbiBgSW5qZWN0b3JEZWZUeXBlYCB3aXRoIGFuXG4gKiBhc3NvY2lhdGVkIGFycmF5IG9mIHByb3ZpZGVycy5cbiAqXG4gKiBPYmplY3RzIG9mIHRoaXMgdHlwZSBjYW4gYmUgbGlzdGVkIGluIHRoZSBpbXBvcnRzIHNlY3Rpb24gb2YgYW4gYEluamVjdG9yRGVmYC5cbiAqXG4gKiBOT1RFOiBUaGlzIGlzIGEgcHJpdmF0ZSB0eXBlIGFuZCBzaG91bGQgbm90IGJlIGV4cG9ydGVkXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW5qZWN0b3JUeXBlV2l0aFByb3ZpZGVyczxUPiB7XG4gIG5nTW9kdWxlOiBJbmplY3RvclR5cGU8VD47XG4gIHByb3ZpZGVycz86IChUeXBlPGFueT58VmFsdWVQcm92aWRlcnxFeGlzdGluZ1Byb3ZpZGVyfEZhY3RvcnlQcm92aWRlcnxDb25zdHJ1Y3RvclByb3ZpZGVyfFxuICAgICAgICAgICAgICAgU3RhdGljQ2xhc3NQcm92aWRlcnxDbGFzc1Byb3ZpZGVyfGFueVtdKVtdO1xufVxuXG5cbi8qKlxuICogQ29uc3RydWN0IGFuIGBJbmplY3RhYmxlRGVmYCB3aGljaCBkZWZpbmVzIGhvdyBhIHRva2VuIHdpbGwgYmUgY29uc3RydWN0ZWQgYnkgdGhlIERJIHN5c3RlbSwgYW5kXG4gKiBpbiB3aGljaCBpbmplY3RvcnMgKGlmIGFueSkgaXQgd2lsbCBiZSBhdmFpbGFibGUuXG4gKlxuICogVGhpcyBzaG91bGQgYmUgYXNzaWduZWQgdG8gYSBzdGF0aWMgYMm1cHJvdmAgZmllbGQgb24gYSB0eXBlLCB3aGljaCB3aWxsIHRoZW4gYmUgYW5cbiAqIGBJbmplY3RhYmxlVHlwZWAuXG4gKlxuICogT3B0aW9uczpcbiAqICogYHByb3ZpZGVkSW5gIGRldGVybWluZXMgd2hpY2ggaW5qZWN0b3JzIHdpbGwgaW5jbHVkZSB0aGUgaW5qZWN0YWJsZSwgYnkgZWl0aGVyIGFzc29jaWF0aW5nIGl0XG4gKiAgIHdpdGggYW4gYEBOZ01vZHVsZWAgb3Igb3RoZXIgYEluamVjdG9yVHlwZWAsIG9yIGJ5IHNwZWNpZnlpbmcgdGhhdCB0aGlzIGluamVjdGFibGUgc2hvdWxkIGJlXG4gKiAgIHByb3ZpZGVkIGluIHRoZSBgJ3Jvb3QnYCBpbmplY3Rvciwgd2hpY2ggd2lsbCBiZSB0aGUgYXBwbGljYXRpb24tbGV2ZWwgaW5qZWN0b3IgaW4gbW9zdCBhcHBzLlxuICogKiBgZmFjdG9yeWAgZ2l2ZXMgdGhlIHplcm8gYXJndW1lbnQgZnVuY3Rpb24gd2hpY2ggd2lsbCBjcmVhdGUgYW4gaW5zdGFuY2Ugb2YgdGhlIGluamVjdGFibGUuXG4gKiAgIFRoZSBmYWN0b3J5IGNhbiBjYWxsIGBpbmplY3RgIHRvIGFjY2VzcyB0aGUgYEluamVjdG9yYCBhbmQgcmVxdWVzdCBpbmplY3Rpb24gb2YgZGVwZW5kZW5jaWVzLlxuICpcbiAqIEBjb2RlR2VuQXBpXG4gKiBAcHVibGljQXBpIFRoaXMgaW5zdHJ1Y3Rpb24gaGFzIGJlZW4gZW1pdHRlZCBieSBWaWV3RW5naW5lIGZvciBzb21lIHRpbWUgYW5kIGlzIGRlcGxveWVkIHRvIG5wbS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIMm1ybVkZWZpbmVJbmplY3RhYmxlPFQ+KG9wdHM6IHtcbiAgdG9rZW46IHVua25vd24sXG4gIHByb3ZpZGVkSW4/OiBUeXBlPGFueT58J3Jvb3QnfCdwbGF0Zm9ybSd8J2FueSd8bnVsbCwgZmFjdG9yeTogKCkgPT4gVCxcbn0pOiBuZXZlciB7XG4gIHJldHVybiAoe1xuICAgICAgICAgICB0b2tlbjogb3B0cy50b2tlbixcbiAgICAgICAgICAgcHJvdmlkZWRJbjogb3B0cy5wcm92aWRlZEluIGFzIGFueSB8fCBudWxsLFxuICAgICAgICAgICBmYWN0b3J5OiBvcHRzLmZhY3RvcnksXG4gICAgICAgICAgIHZhbHVlOiB1bmRlZmluZWQsXG4gICAgICAgICB9IGFzIMm1ybVJbmplY3RhYmxlRGVmPFQ+KSBhcyBuZXZlcjtcbn1cblxuLyoqXG4gKiBAZGVwcmVjYXRlZCBpbiB2OCwgZGVsZXRlIGFmdGVyIHYxMC4gVGhpcyBBUEkgc2hvdWxkIGJlIHVzZWQgb25seSBiZSBnZW5lcmF0ZWQgY29kZSwgYW5kIHRoYXRcbiAqIGNvZGUgc2hvdWxkIG5vdyB1c2UgybXJtWRlZmluZUluamVjdGFibGUgaW5zdGVhZC5cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGNvbnN0IGRlZmluZUluamVjdGFibGUgPSDJtcm1ZGVmaW5lSW5qZWN0YWJsZTtcblxuLyoqXG4gKiBDb25zdHJ1Y3QgYW4gYEluamVjdG9yRGVmYCB3aGljaCBjb25maWd1cmVzIGFuIGluamVjdG9yLlxuICpcbiAqIFRoaXMgc2hvdWxkIGJlIGFzc2lnbmVkIHRvIGEgc3RhdGljIGluamVjdG9yIGRlZiAoYMm1aW5qYCkgZmllbGQgb24gYSB0eXBlLCB3aGljaCB3aWxsIHRoZW4gYmUgYW5cbiAqIGBJbmplY3RvclR5cGVgLlxuICpcbiAqIE9wdGlvbnM6XG4gKlxuICogKiBgZmFjdG9yeWA6IGFuIGBJbmplY3RvclR5cGVgIGlzIGFuIGluc3RhbnRpYWJsZSB0eXBlLCBzbyBhIHplcm8gYXJndW1lbnQgYGZhY3RvcnlgIGZ1bmN0aW9uIHRvXG4gKiAgIGNyZWF0ZSB0aGUgdHlwZSBtdXN0IGJlIHByb3ZpZGVkLiBJZiB0aGF0IGZhY3RvcnkgZnVuY3Rpb24gbmVlZHMgdG8gaW5qZWN0IGFyZ3VtZW50cywgaXQgY2FuXG4gKiAgIHVzZSB0aGUgYGluamVjdGAgZnVuY3Rpb24uXG4gKiAqIGBwcm92aWRlcnNgOiBhbiBvcHRpb25hbCBhcnJheSBvZiBwcm92aWRlcnMgdG8gYWRkIHRvIHRoZSBpbmplY3Rvci4gRWFjaCBwcm92aWRlciBtdXN0XG4gKiAgIGVpdGhlciBoYXZlIGEgZmFjdG9yeSBvciBwb2ludCB0byBhIHR5cGUgd2hpY2ggaGFzIGEgYMm1cHJvdmAgc3RhdGljIHByb3BlcnR5ICh0aGVcbiAqICAgdHlwZSBtdXN0IGJlIGFuIGBJbmplY3RhYmxlVHlwZWApLlxuICogKiBgaW1wb3J0c2A6IGFuIG9wdGlvbmFsIGFycmF5IG9mIGltcG9ydHMgb2Ygb3RoZXIgYEluamVjdG9yVHlwZWBzIG9yIGBJbmplY3RvclR5cGVXaXRoTW9kdWxlYHNcbiAqICAgd2hvc2UgcHJvdmlkZXJzIHdpbGwgYWxzbyBiZSBhZGRlZCB0byB0aGUgaW5qZWN0b3IuIExvY2FsbHkgcHJvdmlkZWQgdHlwZXMgd2lsbCBvdmVycmlkZVxuICogICBwcm92aWRlcnMgZnJvbSBpbXBvcnRzLlxuICpcbiAqIEBjb2RlR2VuQXBpXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiDJtcm1ZGVmaW5lSW5qZWN0b3Iob3B0aW9uczoge2ZhY3Rvcnk6ICgpID0+IGFueSwgcHJvdmlkZXJzPzogYW55W10sIGltcG9ydHM/OiBhbnlbXX0pOlxuICAgIG5ldmVyIHtcbiAgcmV0dXJuICh7XG4gICAgICAgICAgIGZhY3Rvcnk6IG9wdGlvbnMuZmFjdG9yeSxcbiAgICAgICAgICAgcHJvdmlkZXJzOiBvcHRpb25zLnByb3ZpZGVycyB8fCBbXSxcbiAgICAgICAgICAgaW1wb3J0czogb3B0aW9ucy5pbXBvcnRzIHx8IFtdLFxuICAgICAgICAgfSBhcyDJtcm1SW5qZWN0b3JEZWY8YW55PikgYXMgbmV2ZXI7XG59XG5cbi8qKlxuICogUmVhZCB0aGUgaW5qZWN0YWJsZSBkZWYgKGDJtXByb3ZgKSBmb3IgYHR5cGVgIGluIGEgd2F5IHdoaWNoIGlzIGltbXVuZSB0byBhY2NpZGVudGFsbHkgcmVhZGluZ1xuICogaW5oZXJpdGVkIHZhbHVlLlxuICpcbiAqIEBwYXJhbSB0eXBlIEEgdHlwZSB3aGljaCBtYXkgaGF2ZSBpdHMgb3duIChub24taW5oZXJpdGVkKSBgybVwcm92YC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldEluamVjdGFibGVEZWY8VD4odHlwZTogYW55KTogybXJtUluamVjdGFibGVEZWY8VD58bnVsbCB7XG4gIHJldHVybiBnZXRPd25EZWZpbml0aW9uKHR5cGUsIHR5cGVbTkdfUFJPVl9ERUZdKSB8fFxuICAgICAgZ2V0T3duRGVmaW5pdGlvbih0eXBlLCB0eXBlW05HX0lOSkVDVEFCTEVfREVGXSk7XG59XG5cbi8qKlxuICogUmV0dXJuIGBkZWZgIG9ubHkgaWYgaXQgaXMgZGVmaW5lZCBkaXJlY3RseSBvbiBgdHlwZWAgYW5kIGlzIG5vdCBpbmhlcml0ZWQgZnJvbSBhIGJhc2VcbiAqIGNsYXNzIG9mIGB0eXBlYC5cbiAqXG4gKiBUaGUgZnVuY3Rpb24gYE9iamVjdC5oYXNPd25Qcm9wZXJ0eWAgaXMgbm90IHN1ZmZpY2llbnQgdG8gZGlzdGluZ3Vpc2ggdGhpcyBjYXNlIGJlY2F1c2UgaW4gb2xkZXJcbiAqIGJyb3dzZXJzIChlLmcuIElFMTApIHN0YXRpYyBwcm9wZXJ0eSBpbmhlcml0YW5jZSBpcyBpbXBsZW1lbnRlZCBieSBjb3B5aW5nIHRoZSBwcm9wZXJ0aWVzLlxuICpcbiAqIEluc3RlYWQsIHRoZSBkZWZpbml0aW9uJ3MgYHRva2VuYCBpcyBjb21wYXJlZCB0byB0aGUgYHR5cGVgLCBhbmQgaWYgdGhleSBkb24ndCBtYXRjaCB0aGVuIHRoZVxuICogcHJvcGVydHkgd2FzIG5vdCBkZWZpbmVkIGRpcmVjdGx5IG9uIHRoZSB0eXBlIGl0c2VsZiwgYW5kIHdhcyBsaWtlbHkgaW5oZXJpdGVkLiBUaGUgZGVmaW5pdGlvblxuICogaXMgb25seSByZXR1cm5lZCBpZiB0aGUgYHR5cGVgIG1hdGNoZXMgdGhlIGBkZWYudG9rZW5gLlxuICovXG5mdW5jdGlvbiBnZXRPd25EZWZpbml0aW9uPFQ+KHR5cGU6IGFueSwgZGVmOiDJtcm1SW5qZWN0YWJsZURlZjxUPik6IMm1ybVJbmplY3RhYmxlRGVmPFQ+fG51bGwge1xuICByZXR1cm4gZGVmICYmIGRlZi50b2tlbiA9PT0gdHlwZSA/IGRlZiA6IG51bGw7XG59XG5cbi8qKlxuICogUmVhZCB0aGUgaW5qZWN0YWJsZSBkZWYgKGDJtXByb3ZgKSBmb3IgYHR5cGVgIG9yIHJlYWQgdGhlIGDJtXByb3ZgIGZyb20gb25lIG9mIGl0cyBhbmNlc3RvcnMuXG4gKlxuICogQHBhcmFtIHR5cGUgQSB0eXBlIHdoaWNoIG1heSBoYXZlIGDJtXByb3ZgLCB2aWEgaW5oZXJpdGFuY2UuXG4gKlxuICogQGRlcHJlY2F0ZWQgV2lsbCBiZSByZW1vdmVkIGluIGEgZnV0dXJlIHZlcnNpb24gb2YgQW5ndWxhciwgd2hlcmUgYW4gZXJyb3Igd2lsbCBvY2N1ciBpbiB0aGVcbiAqICAgICBzY2VuYXJpbyBpZiB3ZSBmaW5kIHRoZSBgybVwcm92YCBvbiBhbiBhbmNlc3RvciBvbmx5LlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0SW5oZXJpdGVkSW5qZWN0YWJsZURlZjxUPih0eXBlOiBhbnkpOiDJtcm1SW5qZWN0YWJsZURlZjxUPnxudWxsIHtcbiAgLy8gU2VlIGBqaXQvaW5qZWN0YWJsZS50cyNjb21waWxlSW5qZWN0YWJsZWAgZm9yIGNvbnRleHQgb24gTkdfUFJPVl9ERUZfRkFMTEJBQ0suXG4gIGNvbnN0IGRlZiA9IHR5cGUgJiZcbiAgICAgICh0eXBlW05HX1BST1ZfREVGXSB8fCB0eXBlW05HX0lOSkVDVEFCTEVfREVGXSB8fFxuICAgICAgICh0eXBlW05HX1BST1ZfREVGX0ZBTExCQUNLXSAmJiB0eXBlW05HX1BST1ZfREVGX0ZBTExCQUNLXSgpKSk7XG5cbiAgaWYgKGRlZikge1xuICAgIGNvbnN0IHR5cGVOYW1lID0gZ2V0VHlwZU5hbWUodHlwZSk7XG4gICAgLy8gVE9ETyhGVy0xMzA3KTogUmUtYWRkIG5nRGV2TW9kZSB3aGVuIGNsb3N1cmUgY2FuIGhhbmRsZSBpdFxuICAgIC8vIG5nRGV2TW9kZSAmJlxuICAgIGNvbnNvbGUud2FybihcbiAgICAgICAgYERFUFJFQ0FURUQ6IERJIGlzIGluc3RhbnRpYXRpbmcgYSB0b2tlbiBcIiR7XG4gICAgICAgICAgICB0eXBlTmFtZX1cIiB0aGF0IGluaGVyaXRzIGl0cyBASW5qZWN0YWJsZSBkZWNvcmF0b3IgYnV0IGRvZXMgbm90IHByb3ZpZGUgb25lIGl0c2VsZi5cXG5gICtcbiAgICAgICAgYFRoaXMgd2lsbCBiZWNvbWUgYW4gZXJyb3IgaW4gYSBmdXR1cmUgdmVyc2lvbiBvZiBBbmd1bGFyLiBQbGVhc2UgYWRkIEBJbmplY3RhYmxlKCkgdG8gdGhlIFwiJHtcbiAgICAgICAgICAgIHR5cGVOYW1lfVwiIGNsYXNzLmApO1xuICAgIHJldHVybiBkZWY7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cbn1cblxuLyoqIEdldHMgdGhlIG5hbWUgb2YgYSB0eXBlLCBhY2NvdW50aW5nIGZvciBzb21lIGNyb3NzLWJyb3dzZXIgZGlmZmVyZW5jZXMuICovXG5mdW5jdGlvbiBnZXRUeXBlTmFtZSh0eXBlOiBhbnkpOiBzdHJpbmcge1xuICAvLyBgRnVuY3Rpb24ucHJvdG90eXBlLm5hbWVgIGJlaGF2ZXMgZGlmZmVyZW50bHkgYmV0d2VlbiBJRSBhbmQgb3RoZXIgYnJvd3NlcnMuIEluIG1vc3QgYnJvd3NlcnNcbiAgLy8gaXQnbGwgYWx3YXlzIHJldHVybiB0aGUgbmFtZSBvZiB0aGUgZnVuY3Rpb24gaXRzZWxmLCBubyBtYXR0ZXIgaG93IG1hbnkgb3RoZXIgZnVuY3Rpb25zIGl0XG4gIC8vIGluaGVyaXRzIGZyb20uIE9uIElFIHRoZSBmdW5jdGlvbiBkb2Vzbid0IGhhdmUgaXRzIG93biBgbmFtZWAgcHJvcGVydHksIGJ1dCBpdCB0YWtlcyBpdCBmcm9tXG4gIC8vIHRoZSBsb3dlc3QgbGV2ZWwgaW4gdGhlIHByb3RvdHlwZSBjaGFpbi4gRS5nLiBpZiB3ZSBoYXZlIGBjbGFzcyBGb28gZXh0ZW5kcyBQYXJlbnRgIG1vc3RcbiAgLy8gYnJvd3NlcnMgd2lsbCBldmFsdWF0ZSBgRm9vLm5hbWVgIHRvIGBGb29gIHdoaWxlIElFIHdpbGwgcmV0dXJuIGBQYXJlbnRgLiBXZSB3b3JrIGFyb3VuZFxuICAvLyB0aGUgaXNzdWUgYnkgY29udmVydGluZyB0aGUgZnVuY3Rpb24gdG8gYSBzdHJpbmcgYW5kIHBhcnNpbmcgaXRzIG5hbWUgb3V0IHRoYXQgd2F5IHZpYSBhIHJlZ2V4LlxuICBpZiAodHlwZS5oYXNPd25Qcm9wZXJ0eSgnbmFtZScpKSB7XG4gICAgcmV0dXJuIHR5cGUubmFtZTtcbiAgfVxuXG4gIGNvbnN0IG1hdGNoID0gKCcnICsgdHlwZSkubWF0Y2goL15mdW5jdGlvblxccyooW15cXHMoXSspLyk7XG4gIHJldHVybiBtYXRjaCA9PT0gbnVsbCA/ICcnIDogbWF0Y2hbMV07XG59XG5cbi8qKlxuICogUmVhZCB0aGUgaW5qZWN0b3IgZGVmIHR5cGUgaW4gYSB3YXkgd2hpY2ggaXMgaW1tdW5lIHRvIGFjY2lkZW50YWxseSByZWFkaW5nIGluaGVyaXRlZCB2YWx1ZS5cbiAqXG4gKiBAcGFyYW0gdHlwZSB0eXBlIHdoaWNoIG1heSBoYXZlIGFuIGluamVjdG9yIGRlZiAoYMm1aW5qYClcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldEluamVjdG9yRGVmPFQ+KHR5cGU6IGFueSk6IMm1ybVJbmplY3RvckRlZjxUPnxudWxsIHtcbiAgcmV0dXJuIHR5cGUgJiYgKHR5cGUuaGFzT3duUHJvcGVydHkoTkdfSU5KX0RFRikgfHwgdHlwZS5oYXNPd25Qcm9wZXJ0eShOR19JTkpFQ1RPUl9ERUYpKSA/XG4gICAgICAodHlwZSBhcyBhbnkpW05HX0lOSl9ERUZdIDpcbiAgICAgIG51bGw7XG59XG5cbmV4cG9ydCBjb25zdCBOR19QUk9WX0RFRiA9IGdldENsb3N1cmVTYWZlUHJvcGVydHkoe8m1cHJvdjogZ2V0Q2xvc3VyZVNhZmVQcm9wZXJ0eX0pO1xuZXhwb3J0IGNvbnN0IE5HX0lOSl9ERUYgPSBnZXRDbG9zdXJlU2FmZVByb3BlcnR5KHvJtWluajogZ2V0Q2xvc3VyZVNhZmVQcm9wZXJ0eX0pO1xuXG4vLyBPbiBJRTEwIHByb3BlcnRpZXMgZGVmaW5lZCB2aWEgYGRlZmluZVByb3BlcnR5YCB3b24ndCBiZSBpbmhlcml0ZWQgYnkgY2hpbGQgY2xhc3Nlcyxcbi8vIHdoaWNoIHdpbGwgYnJlYWsgaW5oZXJpdGluZyB0aGUgaW5qZWN0YWJsZSBkZWZpbml0aW9uIGZyb20gYSBncmFuZHBhcmVudCB0aHJvdWdoIGFuXG4vLyB1bmRlY29yYXRlZCBwYXJlbnQgY2xhc3MuIFdlIHdvcmsgYXJvdW5kIGl0IGJ5IGRlZmluaW5nIGEgZmFsbGJhY2sgbWV0aG9kIHdoaWNoIHdpbGwgYmVcbi8vIHVzZWQgdG8gcmV0cmlldmUgdGhlIGRlZmluaXRpb24uIFRoaXMgc2hvdWxkIG9ubHkgYmUgYSBwcm9ibGVtIGluIEpJVCBtb2RlLCBiZWNhdXNlIGluXG4vLyBBT1QgVHlwZVNjcmlwdCBzZWVtcyB0byBoYXZlIGEgd29ya2Fyb3VuZCBmb3Igc3RhdGljIHByb3BlcnRpZXMuIFdoZW4gaW5oZXJpdGluZyBmcm9tIGFuXG4vLyB1bmRlY29yYXRlZCBwYXJlbnQgaXMgbm8gbG9uZ2VyIHN1cHBvcnRlZCBpbiB2MTAsIHRoaXMgY2FuIHNhZmVseSBiZSByZW1vdmVkLlxuZXhwb3J0IGNvbnN0IE5HX1BST1ZfREVGX0ZBTExCQUNLID0gZ2V0Q2xvc3VyZVNhZmVQcm9wZXJ0eSh7ybVwcm92RmFsbGJhY2s6IGdldENsb3N1cmVTYWZlUHJvcGVydHl9KTtcblxuLy8gV2UgbmVlZCB0byBrZWVwIHRoZXNlIGFyb3VuZCBzbyB3ZSBjYW4gcmVhZCBvZmYgb2xkIGRlZnMgaWYgbmV3IGRlZnMgYXJlIHVuYXZhaWxhYmxlXG5leHBvcnQgY29uc3QgTkdfSU5KRUNUQUJMRV9ERUYgPSBnZXRDbG9zdXJlU2FmZVByb3BlcnR5KHtuZ0luamVjdGFibGVEZWY6IGdldENsb3N1cmVTYWZlUHJvcGVydHl9KTtcbmV4cG9ydCBjb25zdCBOR19JTkpFQ1RPUl9ERUYgPSBnZXRDbG9zdXJlU2FmZVByb3BlcnR5KHtuZ0luamVjdG9yRGVmOiBnZXRDbG9zdXJlU2FmZVByb3BlcnR5fSk7XG4iXX0=