/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { StaticSymbol } from '../aot/static_symbol';
import * as o from '../output/output_ast';
/**
 * Convert an object map with `Expression` values into a `LiteralMapExpr`.
 */
export function mapToMapExpression(map) {
    const result = Object.keys(map).map(key => ({
        key,
        // The assertion here is because really TypeScript doesn't allow us to express that if the
        // key is present, it will have a value, but this is true in reality.
        value: map[key],
        quoted: false,
    }));
    return o.literalMap(result);
}
/**
 * Convert metadata into an `Expression` in the given `OutputContext`.
 *
 * This operation will handle arrays, references to symbols, or literal `null` or `undefined`.
 */
export function convertMetaToOutput(meta, ctx) {
    if (Array.isArray(meta)) {
        return o.literalArr(meta.map(entry => convertMetaToOutput(entry, ctx)));
    }
    if (meta instanceof StaticSymbol) {
        return ctx.importExpr(meta);
    }
    if (meta == null) {
        return o.literal(meta);
    }
    throw new Error(`Internal error: Unsupported or unknown metadata: ${meta}`);
}
export function typeWithParameters(type, numParams) {
    if (numParams === 0) {
        return o.expressionType(type);
    }
    const params = [];
    for (let i = 0; i < numParams; i++) {
        params.push(o.DYNAMIC_TYPE);
    }
    return o.expressionType(type, undefined, params);
}
const ANIMATE_SYMBOL_PREFIX = '@';
export function prepareSyntheticPropertyName(name) {
    return `${ANIMATE_SYMBOL_PREFIX}${name}`;
}
export function prepareSyntheticListenerName(name, phase) {
    return `${ANIMATE_SYMBOL_PREFIX}${name}.${phase}`;
}
export function isSyntheticPropertyOrListener(name) {
    return name.charAt(0) == ANIMATE_SYMBOL_PREFIX;
}
export function getSyntheticPropertyName(name) {
    // this will strip out listener phase values...
    // @foo.start => @foo
    const i = name.indexOf('.');
    name = i > 0 ? name.substring(0, i) : name;
    if (name.charAt(0) !== ANIMATE_SYMBOL_PREFIX) {
        name = ANIMATE_SYMBOL_PREFIX + name;
    }
    return name;
}
export function prepareSyntheticListenerFunctionName(name, phase) {
    return `animation_${name}_${phase}`;
}
export function jitOnlyGuardedExpression(expr) {
    const ngJitMode = new o.ExternalExpr({ name: 'ngJitMode', moduleName: null });
    const jitFlagNotDefined = new o.BinaryOperatorExpr(o.BinaryOperator.Identical, new o.TypeofExpr(ngJitMode), o.literal('undefined'));
    const jitFlagUndefinedOrTrue = new o.BinaryOperatorExpr(o.BinaryOperator.Or, jitFlagNotDefined, ngJitMode, /* type */ undefined, 
    /* sourceSpan */ undefined, true);
    return new o.BinaryOperatorExpr(o.BinaryOperator.And, jitFlagUndefinedOrTrue, expr);
}
export function wrapReference(value) {
    const wrapped = new o.WrappedNodeExpr(value);
    return { value: wrapped, type: wrapped };
}
//# sourceMappingURL=data:application/json;base64,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