/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/transformers/patch_alias_reference_resolution", ["require", "exports", "tslib", "typescript"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.isAliasImportDeclaration = exports.patchAliasReferenceResolutionOrDie = void 0;
    var tslib_1 = require("tslib");
    var ts = require("typescript");
    /**
     * Patches the alias declaration reference resolution for a given transformation context
     * so that TypeScript knows about the specified alias declarations being referenced.
     *
     * This exists because TypeScript performs analysis of import usage before transformers
     * run and doesn't refresh its state after transformations. This means that imports
     * for symbols used as constructor types are elided due to their original type-only usage.
     *
     * In reality though, since we downlevel decorators and constructor parameters, we want
     * these symbols to be retained in the JavaScript output as they will be used as values
     * at runtime. We can instruct TypeScript to preserve imports for such identifiers by
     * creating a mutable clone of a given import specifier/clause or namespace, but that
     * has the downside of preserving the full import in the JS output. See:
     * https://github.com/microsoft/TypeScript/blob/3eaa7c65f6f076a08a5f7f1946fd0df7c7430259/src/compiler/transformers/ts.ts#L242-L250.
     *
     * This is a trick the CLI used in the past  for constructor parameter downleveling in JIT:
     * https://github.com/angular/angular-cli/blob/b3f84cc5184337666ce61c07b7b9df418030106f/packages/ngtools/webpack/src/transformers/ctor-parameters.ts#L323-L325
     * The trick is not ideal though as it preserves the full import (as outlined before), and it
     * results in a slow-down due to the type checker being involved multiple times. The CLI
     * worked around this import preserving issue by having another complex post-process step that
     * detects and elides unused imports. Note that these unused imports could cause unused chunks
     * being generated by Webpack if the application or library is not marked as side-effect free.
     *
     * This is not ideal though, as we basically re-implement the complex import usage resolution
     * from TypeScript. We can do better by letting TypeScript do the import eliding, but providing
     * information about the alias declarations (e.g. import specifiers) that should not be elided
     * because they are actually referenced (as they will now appear in static properties).
     *
     * More information about these limitations with transformers can be found in:
     *   1. https://github.com/Microsoft/TypeScript/issues/17552.
     *   2. https://github.com/microsoft/TypeScript/issues/17516.
     *   3. https://github.com/angular/tsickle/issues/635.
     *
     * The patch we apply to tell TypeScript about actual referenced aliases (i.e. imported symbols),
     * matches conceptually with the logic that runs internally in TypeScript when the
     * `emitDecoratorMetadata` flag is enabled. TypeScript basically surfaces the same problem and
     * solves it conceptually the same way, but obviously doesn't need to access an `@internal` API.
     *
     * See below. Note that this uses sourcegraph as the TypeScript checker file doesn't display on
     * Github.
     * https://sourcegraph.com/github.com/microsoft/TypeScript@3eaa7c65f6f076a08a5f7f1946fd0df7c7430259/-/blob/src/compiler/checker.ts#L31219-31257
     */
    function patchAliasReferenceResolutionOrDie(context, referencedAliases) {
        // If the `getEmitResolver` method is not available, TS most likely changed the
        // internal structure of the transformation context. We will abort gracefully.
        if (!isTransformationContextWithEmitResolver(context)) {
            throwIncompatibleTransformationContextError();
            return;
        }
        var emitResolver = context.getEmitResolver();
        var originalReferenceResolution = emitResolver.isReferencedAliasDeclaration;
        // If the emit resolver does not have a function called `isReferencedAliasDeclaration`, then
        // we abort gracefully as most likely TS changed the internal structure of the emit resolver.
        if (originalReferenceResolution === undefined) {
            throwIncompatibleTransformationContextError();
            return;
        }
        emitResolver.isReferencedAliasDeclaration = function (node) {
            var args = [];
            for (var _i = 1; _i < arguments.length; _i++) {
                args[_i - 1] = arguments[_i];
            }
            if (isAliasImportDeclaration(node) && referencedAliases.has(node)) {
                return true;
            }
            return originalReferenceResolution.call.apply(originalReferenceResolution, tslib_1.__spread([emitResolver, node], args));
        };
    }
    exports.patchAliasReferenceResolutionOrDie = patchAliasReferenceResolutionOrDie;
    /**
     * Gets whether a given node corresponds to an import alias declaration. Alias
     * declarations can be import specifiers, namespace imports or import clauses
     * as these do not declare an actual symbol but just point to a target declaration.
     */
    function isAliasImportDeclaration(node) {
        return ts.isImportSpecifier(node) || ts.isNamespaceImport(node) || ts.isImportClause(node);
    }
    exports.isAliasImportDeclaration = isAliasImportDeclaration;
    /** Whether the transformation context exposes its emit resolver. */
    function isTransformationContextWithEmitResolver(context) {
        return context.getEmitResolver !== undefined;
    }
    /**
     * Throws an error about an incompatible TypeScript version for which the alias
     * declaration reference resolution could not be monkey-patched. The error will
     * also propose potential solutions that can be applied by developers.
     */
    function throwIncompatibleTransformationContextError() {
        throw Error('Unable to downlevel Angular decorators due to an incompatible TypeScript ' +
            'version.\nIf you recently updated TypeScript and this issue surfaces now, consider ' +
            'downgrading.\n\n' +
            'Please report an issue on the Angular repositories when this issue ' +
            'surfaces and you are using a supposedly compatible TypeScript version.');
    }
});
//# sourceMappingURL=data:application/json;base64,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