/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/transformers/node_emitter_transform", ["require", "exports", "tslib", "typescript", "@angular/compiler-cli/src/transformers/node_emitter", "@angular/compiler-cli/src/transformers/util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.getAngularEmitterTransformFactory = void 0;
    var tslib_1 = require("tslib");
    var ts = require("typescript");
    var node_emitter_1 = require("@angular/compiler-cli/src/transformers/node_emitter");
    var util_1 = require("@angular/compiler-cli/src/transformers/util");
    function getPreamble(original) {
        return "*\n * @fileoverview This file was generated by the Angular template compiler. Do not edit.\n * " + original + "\n * @suppress {suspiciousCode,uselessCode,missingProperties,missingOverride,checkTypes,extraRequire}\n * tslint:disable\n ";
    }
    /**
     * Returns a transformer that does two things for generated files (ngfactory etc):
     * - adds a fileoverview JSDoc comment containing Closure Compiler specific "suppress"ions in JSDoc.
     *   The new comment will contain any fileoverview comment text from the original source file this
     *   file was generated from.
     * - updates generated files that are not in the given map of generatedFiles to have an empty
     *   list of statements as their body.
     */
    function getAngularEmitterTransformFactory(generatedFiles, program, annotateForClosureCompiler) {
        return function () {
            var emitter = new node_emitter_1.TypeScriptNodeEmitter(annotateForClosureCompiler);
            return function (sourceFile) {
                var g = generatedFiles.get(sourceFile.fileName);
                var orig = g && program.getSourceFile(g.srcFileUrl);
                var originalComment = '';
                if (orig)
                    originalComment = getFileoverviewComment(orig);
                var preamble = getPreamble(originalComment);
                if (g && g.stmts) {
                    var _a = tslib_1.__read(emitter.updateSourceFile(sourceFile, g.stmts, preamble), 1), newSourceFile = _a[0];
                    return newSourceFile;
                }
                else if (util_1.GENERATED_FILES.test(sourceFile.fileName)) {
                    // The file should be empty, but emitter.updateSourceFile would still add imports
                    // and various minutiae.
                    // Clear out the source file entirely, only including the preamble comment, so that
                    // ngc produces an empty .js file.
                    var commentStmt = ts.createNotEmittedStatement(sourceFile);
                    ts.addSyntheticLeadingComment(commentStmt, ts.SyntaxKind.MultiLineCommentTrivia, preamble, 
                    /* hasTrailingNewline */ true);
                    return ts.updateSourceFileNode(sourceFile, [commentStmt]);
                }
                return sourceFile;
            };
        };
    }
    exports.getAngularEmitterTransformFactory = getAngularEmitterTransformFactory;
    /**
     * Parses and returns the comment text (without start and end markers) of a \@fileoverview comment
     * in the given source file. Returns the empty string if no such comment can be found.
     */
    function getFileoverviewComment(sourceFile) {
        var trivia = sourceFile.getFullText().substring(0, sourceFile.getStart());
        var leadingComments = ts.getLeadingCommentRanges(trivia, 0);
        if (!leadingComments || leadingComments.length === 0)
            return '';
        var comment = leadingComments[0];
        if (comment.kind !== ts.SyntaxKind.MultiLineCommentTrivia)
            return '';
        // Only comments separated with a \n\n from the file contents are considered file-level comments
        // in TypeScript.
        if (sourceFile.getFullText().substring(comment.end, comment.end + 2) !== '\n\n')
            return '';
        var commentText = sourceFile.getFullText().substring(comment.pos, comment.end);
        // Closure Compiler ignores @suppress and similar if the comment contains @license.
        if (commentText.indexOf('@license') !== -1)
            return '';
        // Also remove any leading `* ` from the first line in case it was a JSDOC comment
        return util_1.stripComment(commentText).replace(/^\*\s+/, '');
    }
});
//# sourceMappingURL=data:application/json;base64,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