/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/ngtsc/typecheck/src/oob", ["require", "exports", "tslib", "typescript", "@angular/compiler-cli/src/ngtsc/diagnostics", "@angular/compiler-cli/src/ngtsc/typecheck/diagnostics"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.OutOfBandDiagnosticRecorderImpl = void 0;
    var tslib_1 = require("tslib");
    var ts = require("typescript");
    var diagnostics_1 = require("@angular/compiler-cli/src/ngtsc/diagnostics");
    var diagnostics_2 = require("@angular/compiler-cli/src/ngtsc/typecheck/diagnostics");
    var OutOfBandDiagnosticRecorderImpl = /** @class */ (function () {
        function OutOfBandDiagnosticRecorderImpl(resolver) {
            this.resolver = resolver;
            this._diagnostics = [];
            /**
             * Tracks which `BindingPipe` nodes have already been recorded as invalid, so only one diagnostic
             * is ever produced per node.
             */
            this.recordedPipes = new Set();
        }
        Object.defineProperty(OutOfBandDiagnosticRecorderImpl.prototype, "diagnostics", {
            get: function () {
                return this._diagnostics;
            },
            enumerable: false,
            configurable: true
        });
        OutOfBandDiagnosticRecorderImpl.prototype.missingReferenceTarget = function (templateId, ref) {
            var mapping = this.resolver.getSourceMapping(templateId);
            var value = ref.value.trim();
            var errorMsg = "No directive found with exportAs '" + value + "'.";
            this._diagnostics.push(diagnostics_2.makeTemplateDiagnostic(templateId, mapping, ref.valueSpan || ref.sourceSpan, ts.DiagnosticCategory.Error, diagnostics_1.ngErrorCode(diagnostics_1.ErrorCode.MISSING_REFERENCE_TARGET), errorMsg));
        };
        OutOfBandDiagnosticRecorderImpl.prototype.missingPipe = function (templateId, ast) {
            if (this.recordedPipes.has(ast)) {
                return;
            }
            var mapping = this.resolver.getSourceMapping(templateId);
            var errorMsg = "No pipe found with name '" + ast.name + "'.";
            var sourceSpan = this.resolver.toParseSourceSpan(templateId, ast.nameSpan);
            if (sourceSpan === null) {
                throw new Error("Assertion failure: no SourceLocation found for usage of pipe '" + ast.name + "'.");
            }
            this._diagnostics.push(diagnostics_2.makeTemplateDiagnostic(templateId, mapping, sourceSpan, ts.DiagnosticCategory.Error, diagnostics_1.ngErrorCode(diagnostics_1.ErrorCode.MISSING_PIPE), errorMsg));
            this.recordedPipes.add(ast);
        };
        OutOfBandDiagnosticRecorderImpl.prototype.illegalAssignmentToTemplateVar = function (templateId, assignment, target) {
            var mapping = this.resolver.getSourceMapping(templateId);
            var errorMsg = "Cannot use variable '" + assignment
                .name + "' as the left-hand side of an assignment expression. Template variables are read-only.";
            var sourceSpan = this.resolver.toParseSourceSpan(templateId, assignment.sourceSpan);
            if (sourceSpan === null) {
                throw new Error("Assertion failure: no SourceLocation found for property binding.");
            }
            this._diagnostics.push(diagnostics_2.makeTemplateDiagnostic(templateId, mapping, sourceSpan, ts.DiagnosticCategory.Error, diagnostics_1.ngErrorCode(diagnostics_1.ErrorCode.WRITE_TO_READ_ONLY_VARIABLE), errorMsg, {
                text: "The variable " + assignment.name + " is declared here.",
                span: target.valueSpan || target.sourceSpan,
            }));
        };
        OutOfBandDiagnosticRecorderImpl.prototype.duplicateTemplateVar = function (templateId, variable, firstDecl) {
            var mapping = this.resolver.getSourceMapping(templateId);
            var errorMsg = "Cannot redeclare variable '" + variable.name + "' as it was previously declared elsewhere for the same template.";
            // The allocation of the error here is pretty useless for variables declared in microsyntax,
            // since the sourceSpan refers to the entire microsyntax property, not a span for the specific
            // variable in question.
            //
            // TODO(alxhub): allocate to a tighter span once one is available.
            this._diagnostics.push(diagnostics_2.makeTemplateDiagnostic(templateId, mapping, variable.sourceSpan, ts.DiagnosticCategory.Error, diagnostics_1.ngErrorCode(diagnostics_1.ErrorCode.DUPLICATE_VARIABLE_DECLARATION), errorMsg, {
                text: "The variable '" + firstDecl.name + "' was first declared here.",
                span: firstDecl.sourceSpan,
            }));
        };
        OutOfBandDiagnosticRecorderImpl.prototype.requiresInlineTcb = function (templateId, node) {
            this._diagnostics.push(makeInlineDiagnostic(templateId, diagnostics_1.ErrorCode.INLINE_TCB_REQUIRED, node.name, "This component requires inline template type-checking, which is not supported by the current environment."));
        };
        OutOfBandDiagnosticRecorderImpl.prototype.requiresInlineTypeConstructors = function (templateId, node, directives) {
            var message;
            if (directives.length > 1) {
                message =
                    "This component uses directives which require inline type constructors, which are not supported by the current environment.";
            }
            else {
                message =
                    "This component uses a directive which requires an inline type constructor, which is not supported by the current environment.";
            }
            this._diagnostics.push(makeInlineDiagnostic(templateId, diagnostics_1.ErrorCode.INLINE_TYPE_CTOR_REQUIRED, node.name, message, directives.map(function (dir) { return diagnostics_1.makeRelatedInformation(dir.name, "Requires an inline type constructor."); })));
        };
        return OutOfBandDiagnosticRecorderImpl;
    }());
    exports.OutOfBandDiagnosticRecorderImpl = OutOfBandDiagnosticRecorderImpl;
    function makeInlineDiagnostic(templateId, code, node, messageText, relatedInformation) {
        return tslib_1.__assign(tslib_1.__assign({}, diagnostics_1.makeDiagnostic(code, node, messageText, relatedInformation)), { componentFile: node.getSourceFile(), templateId: templateId });
    }
});
//# sourceMappingURL=data:application/json;base64,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