(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/ngtsc/imports/src/emitter", ["require", "exports", "tslib", "@angular/compiler", "typescript", "@angular/compiler-cli/src/ngtsc/file_system", "@angular/compiler-cli/src/ngtsc/file_system/src/util", "@angular/compiler-cli/src/ngtsc/util/src/typescript", "@angular/compiler-cli/src/ngtsc/imports/src/find_export"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.UnifiedModulesStrategy = exports.RelativePathStrategy = exports.LogicalProjectStrategy = exports.AbsoluteModuleStrategy = exports.LocalIdentifierStrategy = exports.ReferenceEmitter = exports.ImportFlags = void 0;
    var tslib_1 = require("tslib");
    /**
     * @license
     * Copyright Google LLC All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    var compiler_1 = require("@angular/compiler");
    var ts = require("typescript");
    var file_system_1 = require("@angular/compiler-cli/src/ngtsc/file_system");
    var util_1 = require("@angular/compiler-cli/src/ngtsc/file_system/src/util");
    var typescript_1 = require("@angular/compiler-cli/src/ngtsc/util/src/typescript");
    var find_export_1 = require("@angular/compiler-cli/src/ngtsc/imports/src/find_export");
    /**
     * Flags which alter the imports generated by the `ReferenceEmitter`.
     */
    var ImportFlags;
    (function (ImportFlags) {
        ImportFlags[ImportFlags["None"] = 0] = "None";
        /**
         * Force the generation of a new import when generating a reference, even if an identifier already
         * exists in the target file which could be used instead.
         *
         * This is sometimes required if there's a risk TypeScript might remove imports during emit.
         */
        ImportFlags[ImportFlags["ForceNewImport"] = 1] = "ForceNewImport";
        /**
         * Don't make use of any aliasing information when emitting a reference.
         *
         * This is sometimes required if emitting into a context where generated references will be fed
         * into TypeScript and type-checked (such as in template type-checking).
         */
        ImportFlags[ImportFlags["NoAliasing"] = 2] = "NoAliasing";
        /**
         * Indicates that an import to a type-only declaration is allowed.
         *
         * For references that occur in type-positions, the referred declaration may be a type-only
         * declaration that is not retained during emit. Including this flag allows to emit references to
         * type-only declarations as used in e.g. template type-checking.
         */
        ImportFlags[ImportFlags["AllowTypeImports"] = 4] = "AllowTypeImports";
    })(ImportFlags = exports.ImportFlags || (exports.ImportFlags = {}));
    /**
     * Generates `Expression`s which refer to `Reference`s in a given context.
     *
     * A `ReferenceEmitter` uses one or more `ReferenceEmitStrategy` implementations to produce an
     * `Expression` which refers to a `Reference` in the context of a particular file.
     */
    var ReferenceEmitter = /** @class */ (function () {
        function ReferenceEmitter(strategies) {
            this.strategies = strategies;
        }
        ReferenceEmitter.prototype.emit = function (ref, context, importFlags) {
            var e_1, _a;
            if (importFlags === void 0) { importFlags = ImportFlags.None; }
            try {
                for (var _b = tslib_1.__values(this.strategies), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var strategy = _c.value;
                    var emitted = strategy.emit(ref, context, importFlags);
                    if (emitted !== null) {
                        return emitted;
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            throw new Error("Unable to write a reference to " + typescript_1.nodeNameForError(ref.node) + " in " + ref.node.getSourceFile().fileName + " from " + context.fileName);
        };
        return ReferenceEmitter;
    }());
    exports.ReferenceEmitter = ReferenceEmitter;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations by any local `ts.Identifier`s, if
     * such identifiers are available.
     */
    var LocalIdentifierStrategy = /** @class */ (function () {
        function LocalIdentifierStrategy() {
        }
        LocalIdentifierStrategy.prototype.emit = function (ref, context, importFlags) {
            var refSf = typescript_1.getSourceFile(ref.node);
            // If the emitter has specified ForceNewImport, then LocalIdentifierStrategy should not use a
            // local identifier at all, *except* in the source file where the node is actually declared.
            if (importFlags & ImportFlags.ForceNewImport && refSf !== context) {
                return null;
            }
            // If referenced node is not an actual TS declaration (e.g. `class Foo` or `function foo() {}`,
            // etc) and it is in the current file then just use it directly.
            // This is important because the reference could be a property access (e.g. `exports.foo`). In
            // such a case, the reference's `identities` property would be `[foo]`, which would result in an
            // invalid emission of a free-standing `foo` identifier, rather than `exports.foo`.
            if (!typescript_1.isDeclaration(ref.node) && refSf === context) {
                return new compiler_1.WrappedNodeExpr(ref.node);
            }
            // A Reference can have multiple identities in different files, so it may already have an
            // Identifier in the requested context file.
            var identifier = ref.getIdentityIn(context);
            if (identifier !== null) {
                return new compiler_1.WrappedNodeExpr(identifier);
            }
            else {
                return null;
            }
        };
        return LocalIdentifierStrategy;
    }());
    exports.LocalIdentifierStrategy = LocalIdentifierStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations that come from `node_modules` using
     * an absolute import.
     *
     * Part of this strategy involves looking at the target entry point and identifying the exported
     * name of the targeted declaration, as it might be different from the declared name (e.g. a
     * directive might be declared as FooDirImpl, but exported as FooDir). If no export can be found
     * which maps back to the original directive, an error is thrown.
     */
    var AbsoluteModuleStrategy = /** @class */ (function () {
        function AbsoluteModuleStrategy(program, checker, moduleResolver, reflectionHost) {
            this.program = program;
            this.checker = checker;
            this.moduleResolver = moduleResolver;
            this.reflectionHost = reflectionHost;
            /**
             * A cache of the exports of specific modules, because resolving a module to its exports is a
             * costly operation.
             */
            this.moduleExportsCache = new Map();
        }
        AbsoluteModuleStrategy.prototype.emit = function (ref, context, importFlags) {
            if (ref.bestGuessOwningModule === null) {
                // There is no module name available for this Reference, meaning it was arrived at via a
                // relative path.
                return null;
            }
            else if (!typescript_1.isDeclaration(ref.node)) {
                // It's not possible to import something which isn't a declaration.
                throw new Error("Debug assert: unable to import a Reference to non-declaration of type " + ts.SyntaxKind[ref.node.kind] + ".");
            }
            else if ((importFlags & ImportFlags.AllowTypeImports) === 0 && typescript_1.isTypeDeclaration(ref.node)) {
                throw new Error("Importing a type-only declaration of type " + ts.SyntaxKind[ref.node.kind] + " in a value position is not allowed.");
            }
            // Try to find the exported name of the declaration, if one is available.
            var _a = ref.bestGuessOwningModule, specifier = _a.specifier, resolutionContext = _a.resolutionContext;
            var symbolName = this.resolveImportName(specifier, ref.node, resolutionContext);
            if (symbolName === null) {
                // TODO(alxhub): make this error a ts.Diagnostic pointing at whatever caused this import to be
                // triggered.
                throw new Error("Symbol " + ref.debugName + " declared in " + typescript_1.getSourceFile(ref.node).fileName + " is not exported from " + specifier + " (import into " + context.fileName + ")");
            }
            return new compiler_1.ExternalExpr(new compiler_1.ExternalReference(specifier, symbolName));
        };
        AbsoluteModuleStrategy.prototype.resolveImportName = function (moduleName, target, fromFile) {
            var exports = this.getExportsOfModule(moduleName, fromFile);
            if (exports !== null && exports.has(target)) {
                return exports.get(target);
            }
            else {
                return null;
            }
        };
        AbsoluteModuleStrategy.prototype.getExportsOfModule = function (moduleName, fromFile) {
            if (!this.moduleExportsCache.has(moduleName)) {
                this.moduleExportsCache.set(moduleName, this.enumerateExportsOfModule(moduleName, fromFile));
            }
            return this.moduleExportsCache.get(moduleName);
        };
        AbsoluteModuleStrategy.prototype.enumerateExportsOfModule = function (specifier, fromFile) {
            // First, resolve the module specifier to its entry point, and get the ts.Symbol for it.
            var entryPointFile = this.moduleResolver.resolveModule(specifier, fromFile);
            if (entryPointFile === null) {
                return null;
            }
            var exports = this.reflectionHost.getExportsOfModule(entryPointFile);
            if (exports === null) {
                return null;
            }
            var exportMap = new Map();
            exports.forEach(function (declaration, name) {
                exportMap.set(declaration.node, name);
            });
            return exportMap;
        };
        return AbsoluteModuleStrategy;
    }());
    exports.AbsoluteModuleStrategy = AbsoluteModuleStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations via relative paths, provided they're
     * both in the logical project "space" of paths.
     *
     * This is trickier than it sounds, as the two files may be in different root directories in the
     * project. Simply calculating a file system relative path between the two is not sufficient.
     * Instead, `LogicalProjectPath`s are used.
     */
    var LogicalProjectStrategy = /** @class */ (function () {
        function LogicalProjectStrategy(reflector, logicalFs) {
            this.reflector = reflector;
            this.logicalFs = logicalFs;
        }
        LogicalProjectStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            // Compute the relative path from the importing file to the file being imported. This is done
            // as a logical path computation, because the two files might be in different rootDirs.
            var destPath = this.logicalFs.logicalPathOfSf(destSf);
            if (destPath === null) {
                // The imported file is not within the logical project filesystem.
                return null;
            }
            var originPath = this.logicalFs.logicalPathOfSf(context);
            if (originPath === null) {
                throw new Error("Debug assert: attempt to import from " + context.fileName + " but it's outside the program?");
            }
            // There's no way to emit a relative reference from a file to itself.
            if (destPath === originPath) {
                return null;
            }
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                // The target declaration isn't exported from the file it's declared in. This is an issue!
                return null;
            }
            // With both files expressed as LogicalProjectPaths, getting the module specifier as a relative
            // path is now straightforward.
            var moduleName = file_system_1.LogicalProjectPath.relativePathBetween(originPath, destPath);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return LogicalProjectStrategy;
    }());
    exports.LogicalProjectStrategy = LogicalProjectStrategy;
    /**
     * A `ReferenceEmitStrategy` which constructs relatives paths between `ts.SourceFile`s.
     *
     * This strategy can be used if there is no `rootDir`/`rootDirs` structure for the project which
     * necessitates the stronger logic of `LogicalProjectStrategy`.
     */
    var RelativePathStrategy = /** @class */ (function () {
        function RelativePathStrategy(reflector) {
            this.reflector = reflector;
        }
        RelativePathStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var relativePath = file_system_1.relative(file_system_1.dirname(file_system_1.absoluteFromSourceFile(context)), file_system_1.absoluteFromSourceFile(destSf));
            var moduleName = file_system_1.toRelativeImport(util_1.stripExtension(relativePath));
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return RelativePathStrategy;
    }());
    exports.RelativePathStrategy = RelativePathStrategy;
    /**
     * A `ReferenceEmitStrategy` which uses a `UnifiedModulesHost` to generate absolute import
     * references.
     */
    var UnifiedModulesStrategy = /** @class */ (function () {
        function UnifiedModulesStrategy(reflector, unifiedModulesHost) {
            this.reflector = reflector;
            this.unifiedModulesHost = unifiedModulesHost;
        }
        UnifiedModulesStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                return null;
            }
            var moduleName = this.unifiedModulesHost.fileNameToModuleName(destSf.fileName, context.fileName);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return UnifiedModulesStrategy;
    }());
    exports.UnifiedModulesStrategy = UnifiedModulesStrategy;
});
//# sourceMappingURL=data:application/json;base64,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