"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getWorkspaceConfigGracefully = exports.getTargetTsconfigPath = void 0;
const core_1 = require("@angular-devkit/core");
/** Name of the default Angular CLI workspace configuration files. */
const defaultWorkspaceConfigPaths = ['/angular.json', '/.angular.json'];
/** Gets the tsconfig path from the given target within the specified project. */
function getTargetTsconfigPath(project, targetName) {
    if (project.targets && project.targets[targetName] && project.targets[targetName].options &&
        project.targets[targetName].options.tsConfig) {
        return core_1.normalize(project.targets[targetName].options.tsConfig);
    }
    if (project.architect && project.architect[targetName] && project.architect[targetName].options &&
        project.architect[targetName].options.tsConfig) {
        return core_1.normalize(project.architect[targetName].options.tsConfig);
    }
    return null;
}
exports.getTargetTsconfigPath = getTargetTsconfigPath;
/**
 * Resolve the workspace configuration of the specified tree gracefully. We cannot use the utility
 * functions from the default Angular schematics because those might not be present in older
 * versions of the CLI. Also it's important to resolve the workspace gracefully because
 * the CLI project could be still using `.angular-cli.json` instead of thew new config.
 */
function getWorkspaceConfigGracefully(tree) {
    const path = defaultWorkspaceConfigPaths.find(filePath => tree.exists(filePath));
    const configBuffer = tree.read(path);
    if (!path || !configBuffer) {
        return null;
    }
    try {
        // Parse the workspace file as JSON5 which is also supported for CLI
        // workspace configurations.
        return core_1.parseJson(configBuffer.toString(), core_1.JsonParseMode.Json5);
    }
    catch (e) {
        return null;
    }
}
exports.getWorkspaceConfigGracefully = getWorkspaceConfigGracefully;
//# sourceMappingURL=data:application/json;base64,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