/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { dispatchFakeEvent, dispatchKeyboardEvent } from './dispatch-events';
import { triggerFocus } from './element-focus';
/**
 * Checks whether the given Element is a text input element.
 * @docs-private
 */
export function isTextInput(element) {
    const nodeName = element.nodeName.toLowerCase();
    return nodeName === 'input' || nodeName === 'textarea';
}
export function typeInElement(element, ...modifiersAndKeys) {
    const first = modifiersAndKeys[0];
    let modifiers;
    let rest;
    if (typeof first !== 'string' && first.keyCode === undefined && first.key === undefined) {
        modifiers = first;
        rest = modifiersAndKeys.slice(1);
    }
    else {
        modifiers = {};
        rest = modifiersAndKeys;
    }
    const keys = rest
        .map(k => typeof k === 'string' ?
        k.split('').map(c => ({ keyCode: c.toUpperCase().charCodeAt(0), key: c })) : [k])
        .reduce((arr, k) => arr.concat(k), []);
    triggerFocus(element);
    for (const key of keys) {
        dispatchKeyboardEvent(element, 'keydown', key.keyCode, key.key, modifiers);
        dispatchKeyboardEvent(element, 'keypress', key.keyCode, key.key, modifiers);
        if (isTextInput(element) && key.key && key.key.length === 1) {
            element.value += key.key;
            dispatchFakeEvent(element, 'input');
        }
        dispatchKeyboardEvent(element, 'keyup', key.keyCode, key.key, modifiers);
    }
}
/**
 * Clears the text in an input or textarea element.
 * @docs-private
 */
export function clearElement(element) {
    triggerFocus(element);
    element.value = '';
    dispatchFakeEvent(element, 'input');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZS1pbi1lbGVtZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vc3JjL2Nkay90ZXN0aW5nL3Rlc3RiZWQvZmFrZS1ldmVudHMvdHlwZS1pbi1lbGVtZW50LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQUdILE9BQU8sRUFBQyxpQkFBaUIsRUFBRSxxQkFBcUIsRUFBQyxNQUFNLG1CQUFtQixDQUFDO0FBQzNFLE9BQU8sRUFBQyxZQUFZLEVBQUMsTUFBTSxpQkFBaUIsQ0FBQztBQUU3Qzs7O0dBR0c7QUFDSCxNQUFNLFVBQVUsV0FBVyxDQUFDLE9BQWdCO0lBQzFDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDaEQsT0FBTyxRQUFRLEtBQUssT0FBTyxJQUFJLFFBQVEsS0FBSyxVQUFVLENBQUU7QUFDMUQsQ0FBQztBQXVCRCxNQUFNLFVBQVUsYUFBYSxDQUFDLE9BQW9CLEVBQUUsR0FBRyxnQkFBcUI7SUFDMUUsTUFBTSxLQUFLLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbEMsSUFBSSxTQUF1QixDQUFDO0lBQzVCLElBQUksSUFBbUQsQ0FBQztJQUN4RCxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxLQUFLLENBQUMsT0FBTyxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsR0FBRyxLQUFLLFNBQVMsRUFBRTtRQUN2RixTQUFTLEdBQUcsS0FBSyxDQUFDO1FBQ2xCLElBQUksR0FBRyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDbEM7U0FBTTtRQUNMLFNBQVMsR0FBRyxFQUFFLENBQUM7UUFDZixJQUFJLEdBQUcsZ0JBQWdCLENBQUM7S0FDekI7SUFDRCxNQUFNLElBQUksR0FBdUMsSUFBSTtTQUNoRCxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxRQUFRLENBQUMsQ0FBQztRQUM3QixDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ2xGLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFM0MsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3RCLEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxFQUFFO1FBQ3RCLHFCQUFxQixDQUFDLE9BQU8sRUFBRSxTQUFTLEVBQUUsR0FBRyxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzNFLHFCQUFxQixDQUFDLE9BQU8sRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzVFLElBQUksV0FBVyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEdBQUcsQ0FBQyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNELE9BQU8sQ0FBQyxLQUFLLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQztZQUN6QixpQkFBaUIsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7U0FDckM7UUFDRCxxQkFBcUIsQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLEdBQUcsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztLQUMxRTtBQUNILENBQUM7QUFFRDs7O0dBR0c7QUFDSCxNQUFNLFVBQVUsWUFBWSxDQUFDLE9BQStDO0lBQzFFLFlBQVksQ0FBQyxPQUFzQixDQUFDLENBQUM7SUFDckMsT0FBTyxDQUFDLEtBQUssR0FBRyxFQUFFLENBQUM7SUFDbkIsaUJBQWlCLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0FBQ3RDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIExMQyBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cblxuaW1wb3J0IHtNb2RpZmllcktleXN9IGZyb20gJ0Bhbmd1bGFyL2Nkay90ZXN0aW5nJztcbmltcG9ydCB7ZGlzcGF0Y2hGYWtlRXZlbnQsIGRpc3BhdGNoS2V5Ym9hcmRFdmVudH0gZnJvbSAnLi9kaXNwYXRjaC1ldmVudHMnO1xuaW1wb3J0IHt0cmlnZ2VyRm9jdXN9IGZyb20gJy4vZWxlbWVudC1mb2N1cyc7XG5cbi8qKlxuICogQ2hlY2tzIHdoZXRoZXIgdGhlIGdpdmVuIEVsZW1lbnQgaXMgYSB0ZXh0IGlucHV0IGVsZW1lbnQuXG4gKiBAZG9jcy1wcml2YXRlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc1RleHRJbnB1dChlbGVtZW50OiBFbGVtZW50KTogZWxlbWVudCBpcyBIVE1MSW5wdXRFbGVtZW50IHwgSFRNTFRleHRBcmVhRWxlbWVudCB7XG4gIGNvbnN0IG5vZGVOYW1lID0gZWxlbWVudC5ub2RlTmFtZS50b0xvd2VyQ2FzZSgpO1xuICByZXR1cm4gbm9kZU5hbWUgPT09ICdpbnB1dCcgfHwgbm9kZU5hbWUgPT09ICd0ZXh0YXJlYScgO1xufVxuXG4vKipcbiAqIEZvY3VzZXMgYW4gaW5wdXQsIHNldHMgaXRzIHZhbHVlIGFuZCBkaXNwYXRjaGVzXG4gKiB0aGUgYGlucHV0YCBldmVudCwgc2ltdWxhdGluZyB0aGUgdXNlciB0eXBpbmcuXG4gKiBAcGFyYW0gZWxlbWVudCBFbGVtZW50IG9udG8gd2hpY2ggdG8gc2V0IHRoZSB2YWx1ZS5cbiAqIEBwYXJhbSBrZXlzIFRoZSBrZXlzIHRvIHNlbmQgdG8gdGhlIGVsZW1lbnQuXG4gKiBAZG9jcy1wcml2YXRlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB0eXBlSW5FbGVtZW50KFxuICAgIGVsZW1lbnQ6IEhUTUxFbGVtZW50LCAuLi5rZXlzOiAoc3RyaW5nIHwge2tleUNvZGU/OiBudW1iZXIsIGtleT86IHN0cmluZ30pW10pOiB2b2lkO1xuXG4vKipcbiAqIEZvY3VzZXMgYW4gaW5wdXQsIHNldHMgaXRzIHZhbHVlIGFuZCBkaXNwYXRjaGVzXG4gKiB0aGUgYGlucHV0YCBldmVudCwgc2ltdWxhdGluZyB0aGUgdXNlciB0eXBpbmcuXG4gKiBAcGFyYW0gZWxlbWVudCBFbGVtZW50IG9udG8gd2hpY2ggdG8gc2V0IHRoZSB2YWx1ZS5cbiAqIEBwYXJhbSBtb2RpZmllcnMgTW9kaWZpZXIga2V5cyB0aGF0IGFyZSBoZWxkIHdoaWxlIHR5cGluZy5cbiAqIEBwYXJhbSBrZXlzIFRoZSBrZXlzIHRvIHNlbmQgdG8gdGhlIGVsZW1lbnQuXG4gKiBAZG9jcy1wcml2YXRlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB0eXBlSW5FbGVtZW50KGVsZW1lbnQ6IEhUTUxFbGVtZW50LCBtb2RpZmllcnM6IE1vZGlmaWVyS2V5cyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIC4uLmtleXM6IChzdHJpbmcgfCB7a2V5Q29kZT86IG51bWJlciwga2V5Pzogc3RyaW5nfSlbXSk6IHZvaWQ7XG5cbmV4cG9ydCBmdW5jdGlvbiB0eXBlSW5FbGVtZW50KGVsZW1lbnQ6IEhUTUxFbGVtZW50LCAuLi5tb2RpZmllcnNBbmRLZXlzOiBhbnkpIHtcbiAgY29uc3QgZmlyc3QgPSBtb2RpZmllcnNBbmRLZXlzWzBdO1xuICBsZXQgbW9kaWZpZXJzOiBNb2RpZmllcktleXM7XG4gIGxldCByZXN0OiAoc3RyaW5nIHwge2tleUNvZGU/OiBudW1iZXIsIGtleT86IHN0cmluZ30pW107XG4gIGlmICh0eXBlb2YgZmlyc3QgIT09ICdzdHJpbmcnICYmIGZpcnN0LmtleUNvZGUgPT09IHVuZGVmaW5lZCAmJiBmaXJzdC5rZXkgPT09IHVuZGVmaW5lZCkge1xuICAgIG1vZGlmaWVycyA9IGZpcnN0O1xuICAgIHJlc3QgPSBtb2RpZmllcnNBbmRLZXlzLnNsaWNlKDEpO1xuICB9IGVsc2Uge1xuICAgIG1vZGlmaWVycyA9IHt9O1xuICAgIHJlc3QgPSBtb2RpZmllcnNBbmRLZXlzO1xuICB9XG4gIGNvbnN0IGtleXM6IHtrZXlDb2RlPzogbnVtYmVyLCBrZXk/OiBzdHJpbmd9W10gPSByZXN0XG4gICAgICAubWFwKGsgPT4gdHlwZW9mIGsgPT09ICdzdHJpbmcnID9cbiAgICAgICAgICBrLnNwbGl0KCcnKS5tYXAoYyA9PiAoe2tleUNvZGU6IGMudG9VcHBlckNhc2UoKS5jaGFyQ29kZUF0KDApLCBrZXk6IGN9KSkgOiBba10pXG4gICAgICAucmVkdWNlKChhcnIsIGspID0+IGFyci5jb25jYXQoayksIFtdKTtcblxuICB0cmlnZ2VyRm9jdXMoZWxlbWVudCk7XG4gIGZvciAoY29uc3Qga2V5IG9mIGtleXMpIHtcbiAgICBkaXNwYXRjaEtleWJvYXJkRXZlbnQoZWxlbWVudCwgJ2tleWRvd24nLCBrZXkua2V5Q29kZSwga2V5LmtleSwgbW9kaWZpZXJzKTtcbiAgICBkaXNwYXRjaEtleWJvYXJkRXZlbnQoZWxlbWVudCwgJ2tleXByZXNzJywga2V5LmtleUNvZGUsIGtleS5rZXksIG1vZGlmaWVycyk7XG4gICAgaWYgKGlzVGV4dElucHV0KGVsZW1lbnQpICYmIGtleS5rZXkgJiYga2V5LmtleS5sZW5ndGggPT09IDEpIHtcbiAgICAgIGVsZW1lbnQudmFsdWUgKz0ga2V5LmtleTtcbiAgICAgIGRpc3BhdGNoRmFrZUV2ZW50KGVsZW1lbnQsICdpbnB1dCcpO1xuICAgIH1cbiAgICBkaXNwYXRjaEtleWJvYXJkRXZlbnQoZWxlbWVudCwgJ2tleXVwJywga2V5LmtleUNvZGUsIGtleS5rZXksIG1vZGlmaWVycyk7XG4gIH1cbn1cblxuLyoqXG4gKiBDbGVhcnMgdGhlIHRleHQgaW4gYW4gaW5wdXQgb3IgdGV4dGFyZWEgZWxlbWVudC5cbiAqIEBkb2NzLXByaXZhdGVcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNsZWFyRWxlbWVudChlbGVtZW50OiBIVE1MSW5wdXRFbGVtZW50IHwgSFRNTFRleHRBcmVhRWxlbWVudCkge1xuICB0cmlnZ2VyRm9jdXMoZWxlbWVudCBhcyBIVE1MRWxlbWVudCk7XG4gIGVsZW1lbnQudmFsdWUgPSAnJztcbiAgZGlzcGF0Y2hGYWtlRXZlbnQoZWxlbWVudCwgJ2lucHV0Jyk7XG59XG4iXX0=