<!-- prettier-ignore-start -->

# cns-ratelimit - RateLimiter Module

Install with:

    npm install @cambium/cns-ratelimiter

# Features

### 1. RateLimiter

| .   | Type             | Description                                                |
| --- | ---------------- | ---------------------------------------------------------- |
| 1.  | Fixed Rate       | Fixed Rate Based Rate Limit                                |
| 2.  | Weight Based     | Rate limit can be applied based on System total weight     |
| 3.  | Key based Weight | Rate limit can be applied based on custom Key based weight |
| 4.  | Composite        | Fixed rate can be combined with weight based for any API   |

<br />

### 2. Generic Counter

<br />

# Quick Start with RateLimiter

## Define Fixed Rated RateLimiter

```js
const RateLimiter = require('@cambium/cns-ratelimit').RateLimiter;

var rl = new RateLimiter('rlTest1', TestServer.redisClient, {
  weightSystemEnabled: false
});
```

## Define a Route with Fixed Rate

<!-- prettier-ignore -->
```js
app.route.get(  '/test/x/:id', rl.rateOn((req) => {
  return req.params['id']; // returns target key
}).apply({ rate: '10/5s' }), (req, res) => {
  res.send({ msg: 'ok' });
});

```

## Define a Weight Based RateLimiter

<!-- prettier-ignore -->
```js
var rl = new RateLimiter('rlTest1', TestServer.redisClient, {
  weightSystemEnabled: true,
  totalWeight: 1000
});

app.route.get('/test/x/:id', rl.weightOn().apply({ weight: 12 }), (req, res) => {
  res.send({ msg: 'ok' });
});

```

## Define a Key based Shared Weight RateLimiter

<!-- prettier-ignore -->
```js
var rl = new RateLimiter('api', TestServer.redisClient, {
  weightSystemEnabled: true,
  sharedWeightEnabled: true,
  sharedRateLimiterIds: ['api', 'device', 'web']
});

rl.defineKeyBasedMaxWeight({
  cid1: 20,
  cid2: 40,
  cid3: 60
});

rl.defineKeyBasedMaxSharedWeight({
  cid1: 10,
  cid2: 30,
  cid3: 40
});

app.route.get('/test/x/:cid', rl.weightOn((req) => {
  return 'cid' + req.params['cid']; // returns target key
}).apply({ weight: 12 }), (req, res) => {
  res.send({ msg: 'ok' });
});

```

## Define Key based Shared Weight & Fixed Rate composite RateLimiter

<!-- prettier-ignore -->
```js
var rl = new RateLimiter('api', TestServer.redisClient, {
  weightSystemEnabled: true,
  sharedWeightEnabled: true,
  sharedRateLimiterIds: ['api', 'device', 'web']
});

rl.defineKeyBasedMaxWeight({
  cid1: 20,
  cid2: 40,
  cid3: 60
});

rl.defineKeyBasedMaxSharedWeight({
  cid1: 10,
  cid2: 30,
  cid3: 40
});

app.route.get('/test/x/:cid', rl.weightOn((req) => {
  return 'cid' + req.params['cid']; // returns target key
}).apply({ weight: 12, rate: '100/5m' }), (req, res) => {
  res.send({ msg: 'ok' });
});
```

</br>

# Working with plain Counter

```js
const Counter = require('@cambium/cns-ratelimit').Counter;

var myCounter = new Counter(redisDb, 'maxValueKey', 'currentValueKey');

myCounter.setMaxValue(100);

myCounter.add(10);

myCounter.subtract(10);
```

</br>

## Methods available for RateLimiter

| Option                  | Type                                                                                                           | Returns        | Description                                        |
| ----------------------- | -------------------------------------------------------------------------------------------------------------- | -------------- | -------------------------------------------------- |
| rateOn                  | functon(arg: Function)                                                                                         | Rules Object   | Apply only rate based rate limiter                 |
| weightOn                | functon(arg: Function)                                                                                         | Rules Object   | Apply only weight based rate limiter               |
| rateAndWeightOn         | functon(arg: Function)                                                                                         | Rules Object   | Apply rate and weight based composite rate limiter |
| defineKeyBasedMaxWeight | functon(IMaxWeight)</br> Example: </br> rateLimter.**defineKeyBasedMaxWeight**({ cid1:20, cid2: 30, cid3: 50}) | Promise Object | Set Key based Weight                               |

<br />

## Options for RateLimiter of type IRateLimiterConfig

| Option                    | Type                   | Default                                                                                                                                                                                               | Description                                               |
| ------------------------- | ---------------------- | ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------- |
| weightSystemEnabled       | Boolean                | false                                                                                                                                                                                                 | Whether weight based rate limiter enabled                 |
| sharedWeightEnabled       | Boolean                | false                                                                                                                                                                                                 | Whether keybased shared weight based rate limiter enabled |
| sharedRateLimiterIds      | Array<string>          | false                                                                                                                                                                                                 | Define ids of the rateLimiters Ids                        |
| totalWeight               | Number                 | undefined                                                                                                                                                                                             | Define total system weight                                |
| getWeight                 | Number                 | 0                                                                                                                                                                                                     | GET Request weight                                        |
| postWeight                | Number                 | 0                                                                                                                                                                                                     | POST Request weight                                       |
| deleteWeight              | Number                 | 0                                                                                                                                                                                                     | DELETE Request weight                                     |
| putWeight                 | Number                 | 0                                                                                                                                                                                                     | PUT Request weight                                        |
| patchWeight               | Number                 | 0                                                                                                                                                                                                     | PATCH Request weight                                      |
| limitReachedHttpCode      | 429                    | { message: 'Rate limit exceeded' }</br>Also returns the following headers</br> RateLimit-Limit: as total hit limit,</br>RateLimit-Remaining as remaining hit,</br>RateLimit-Reset as remaining expiry | Fixed rate can be combined with weight based for any API  |
| limitReachedResData       | Custom response Object | false                                                                                                                                                                                                 | Http response message on rate limit reached               |
| serviceFailureHttpResCode | 503                    | { message: 'Server is busy'}                                                                                                                                                                          | Http status code on weight limit reached                  |
| serviceFailureResData     | Custom response Object | false                                                                                                                                                                                                 | Http response message on weight limit reached             |
| weightReachedHttpResCode  | Number                 | 500                                                                                                                                                                                                   | Http response code on server failure                      |
| weightReachedHttpResData  | Custom response Object | { message: 'Something went wrong!'}                                                                                                                                                                   | Http response message on system failure                   |
| logger                    | Logger Object          | console                                                                                                                                                                                               | Custom logger instance                                    |

<br />

## Options for _rules.apply()_ API of type IRateLimitConditions

| Option       | Type                     | Default   | Description                                                                                                                                                                                     |
| ------------ | ------------------------ | --------- | ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| rate         | String/Object[id:string] | '100/1m'  | '100/5s' (100 per 5 seconds) </br> '100/5m' (100 per 5 minutes) </br> '100/5h' (100 per 1 hour) </br> '100/5h' (100 per 1 hour) </br> '100/1d' (100 per 1 day) </br> '100/1M' (100 per 1 month) |
| http_method  | Boolean                  | true      | Whether http_header should be considered with weight value as defined in RateLimiter options                                                                                                    |
| http_headers | Array<String>            | undefined | List of headers that will be part of key to uniquely identify the resource and apply rules                                                                                                      |
| weight       | Number                   | 0         | Additional weight to apply for current rule                                                                                                                                                     |

<br />
<!-- prettier-ignore-end -->
