/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
'use strict';
(function (context) {
    var Mocha = context.Mocha;
    if (typeof Mocha === 'undefined') {
        throw new Error('Missing Mocha.js');
    }
    if (typeof Zone === 'undefined') {
        throw new Error('Missing Zone.js');
    }
    var ProxyZoneSpec = Zone['ProxyZoneSpec'];
    var SyncTestZoneSpec = Zone['SyncTestZoneSpec'];
    if (!ProxyZoneSpec) {
        throw new Error('Missing ProxyZoneSpec');
    }
    if (Mocha['__zone_patch__']) {
        throw new Error('"Mocha" has already been patched with "Zone".');
    }
    Mocha['__zone_patch__'] = true;
    var rootZone = Zone.current;
    var syncZone = rootZone.fork(new SyncTestZoneSpec('Mocha.describe'));
    var testZone = null;
    var suiteZone = rootZone.fork(new ProxyZoneSpec());
    var mochaOriginal = {
        after: Mocha.after,
        afterEach: Mocha.afterEach,
        before: Mocha.before,
        beforeEach: Mocha.beforeEach,
        describe: Mocha.describe,
        it: Mocha.it
    };
    function modifyArguments(args, syncTest, asyncTest) {
        var _loop_1 = function (i) {
            var arg = args[i];
            if (typeof arg === 'function') {
                // The `done` callback is only passed through if the function expects at
                // least one argument.
                // Note we have to make a function with correct number of arguments,
                // otherwise mocha will
                // think that all functions are sync or async.
                args[i] = (arg.length === 0) ? syncTest(arg) : asyncTest(arg);
                // Mocha uses toString to view the test body in the result list, make sure we return the
                // correct function body
                args[i].toString = function () { return arg.toString(); };
            }
        };
        for (var i = 0; i < args.length; i++) {
            _loop_1(i);
        }
        return args;
    }
    function wrapDescribeInZone(args) {
        var syncTest = function (fn) {
            return function () { return syncZone.run(fn, this, arguments); };
        };
        return modifyArguments(args, syncTest);
    }
    function wrapTestInZone(args) {
        var asyncTest = function (fn) {
            return function (done) { return testZone.run(fn, this, [done]); };
        };
        var syncTest = function (fn) {
            return function () { return testZone.run(fn, this); };
        };
        return modifyArguments(args, syncTest, asyncTest);
    }
    function wrapSuiteInZone(args) {
        var asyncTest = function (fn) {
            return function (done) { return suiteZone.run(fn, this, [done]); };
        };
        var syncTest = function (fn) {
            return function () { return suiteZone.run(fn, this); };
        };
        return modifyArguments(args, syncTest, asyncTest);
    }
    context.describe = context.suite = Mocha.describe = function () {
        return mochaOriginal.describe.apply(this, wrapDescribeInZone(arguments));
    };
    context.xdescribe = context.suite.skip = Mocha.describe.skip = function () {
        return mochaOriginal.describe.skip.apply(this, wrapDescribeInZone(arguments));
    };
    context.describe.only = context.suite.only = Mocha.describe.only = function () {
        return mochaOriginal.describe.only.apply(this, wrapDescribeInZone(arguments));
    };
    context.it = context.specify = context.test =
        Mocha.it = function () { return mochaOriginal.it.apply(this, wrapTestInZone(arguments)); };
    context.xit = context.xspecify = Mocha.it.skip = function () {
        return mochaOriginal.it.skip.apply(this, wrapTestInZone(arguments));
    };
    context.it.only = context.test.only = Mocha.it.only = function () {
        return mochaOriginal.it.only.apply(this, wrapTestInZone(arguments));
    };
    context.after = context.suiteTeardown = Mocha.after = function () {
        return mochaOriginal.after.apply(this, wrapSuiteInZone(arguments));
    };
    context.afterEach = context.teardown = Mocha.afterEach = function () {
        return mochaOriginal.afterEach.apply(this, wrapTestInZone(arguments));
    };
    context.before = context.suiteSetup = Mocha.before = function () {
        return mochaOriginal.before.apply(this, wrapSuiteInZone(arguments));
    };
    context.beforeEach = context.setup = Mocha.beforeEach = function () {
        return mochaOriginal.beforeEach.apply(this, wrapTestInZone(arguments));
    };
    (function (originalRunTest, originalRun) {
        Mocha.Runner.prototype.runTest = function (fn) {
            var _this = this;
            Zone.current.scheduleMicroTask('mocha.forceTask', function () { originalRunTest.call(_this, fn); });
        };
        Mocha.Runner.prototype.run = function (fn) {
            this.on('test', function (e) { testZone = rootZone.fork(new ProxyZoneSpec()); });
            this.on('fail', function (test, err) {
                var proxyZoneSpec = testZone && testZone.get('ProxyZoneSpec');
                if (proxyZoneSpec && err) {
                    try {
                        // try catch here in case err.message is not writable
                        err.message += proxyZoneSpec.getAndClearPendingTasksInfo();
                    }
                    catch (error) {
                    }
                }
            });
            return originalRun.call(this, fn);
        };
    })(Mocha.Runner.prototype.runTest, Mocha.Runner.prototype.run);
})(typeof window !== 'undefined' && window || typeof self !== 'undefined' && self || global);
//# sourceMappingURL=data:application/json;base64,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