'use strict'

const { range } = require('./util')

/**
 * A.1 Unassigned code points in Unicode 3.2
 * @link https://tools.ietf.org/html/rfc3454#appendix-A.1
 */
const unassigned_code_points = new Set([
  0x0221,
  ...range(0x0234, 0x024F),
  ...range(0x02AE, 0x02AF),
  ...range(0x02EF, 0x02FF),
  ...range(0x0350, 0x035F),
  ...range(0x0370, 0x0373),
  ...range(0x0376, 0x0379),
  ...range(0x037B, 0x037D),
  ...range(0x037F, 0x0383),
  0x038B,
  0x038D,
  0x03A2,
  0x03CF,
  ...range(0x03F7, 0x03FF),
  0x0487,
  0x04CF,
  ...range(0x04F6, 0x04F7),
  ...range(0x04FA, 0x04FF),
  ...range(0x0510, 0x0530),
  ...range(0x0557, 0x0558),
  0x0560,
  0x0588,
  ...range(0x058B, 0x0590),
  0x05A2,
  0x05BA,
  ...range(0x05C5, 0x05CF),
  ...range(0x05EB, 0x05EF),
  ...range(0x05F5, 0x060B),
  ...range(0x060D, 0x061A),
  ...range(0x061C, 0x061E),
  0x0620,
  ...range(0x063B, 0x063F),
  ...range(0x0656, 0x065F),
  ...range(0x06EE, 0x06EF),
  0x06FF,
  0x070E,
  ...range(0x072D, 0x072F),
  ...range(0x074B, 0x077F),
  ...range(0x07B2, 0x0900),
  0x0904,
  ...range(0x093A, 0x093B),
  ...range(0x094E, 0x094F),
  ...range(0x0955, 0x0957),
  ...range(0x0971, 0x0980),
  0x0984,
  ...range(0x098D, 0x098E),
  ...range(0x0991, 0x0992),
  0x09A9,
  0x09B1,
  ...range(0x09B3, 0x09B5),
  ...range(0x09BA, 0x09BB),
  0x09BD,
  ...range(0x09C5, 0x09C6),
  ...range(0x09C9, 0x09CA),
  ...range(0x09CE, 0x09D6),
  ...range(0x09D8, 0x09DB),
  0x09DE,
  ...range(0x09E4, 0x09E5),
  ...range(0x09FB, 0x0A01),
  ...range(0x0A03, 0x0A04),
  ...range(0x0A0B, 0x0A0E),
  ...range(0x0A11, 0x0A12),
  0x0A29,
  0x0A31,
  0x0A34,
  0x0A37,
  ...range(0x0A3A, 0x0A3B),
  0x0A3D,
  ...range(0x0A43, 0x0A46),
  ...range(0x0A49, 0x0A4A),
  ...range(0x0A4E, 0x0A58),
  0x0A5D,
  ...range(0x0A5F, 0x0A65),
  ...range(0x0A75, 0x0A80),
  0x0A84,
  0x0A8C,
  0x0A8E,
  0x0A92,
  0x0AA9,
  0x0AB1,
  0x0AB4,
  ...range(0x0ABA, 0x0ABB),
  0x0AC6,
  0x0ACA,
  ...range(0x0ACE, 0x0ACF),
  ...range(0x0AD1, 0x0ADF),
  ...range(0x0AE1, 0x0AE5),
  ...range(0x0AF0, 0x0B00),
  0x0B04,
  ...range(0x0B0D, 0x0B0E),
  ...range(0x0B11, 0x0B12),
  0x0B29,
  0x0B31,
  ...range(0x0B34, 0x0B35),
  ...range(0x0B3A, 0x0B3B),
  ...range(0x0B44, 0x0B46),
  ...range(0x0B49, 0x0B4A),
  ...range(0x0B4E, 0x0B55),
  ...range(0x0B58, 0x0B5B),
  0x0B5E,
  ...range(0x0B62, 0x0B65),
  ...range(0x0B71, 0x0B81),
  0x0B84,
  ...range(0x0B8B, 0x0B8D),
  0x0B91,
  ...range(0x0B96, 0x0B98),
  0x0B9B,
  0x0B9D,
  ...range(0x0BA0, 0x0BA2),
  ...range(0x0BA5, 0x0BA7),
  ...range(0x0BAB, 0x0BAD),
  0x0BB6,
  ...range(0x0BBA, 0x0BBD),
  ...range(0x0BC3, 0x0BC5),
  0x0BC9,
  ...range(0x0BCE, 0x0BD6),
  ...range(0x0BD8, 0x0BE6),
  ...range(0x0BF3, 0x0C00),
  0x0C04,
  0x0C0D,
  0x0C11,
  0x0C29,
  0x0C34,
  ...range(0x0C3A, 0x0C3D),
  0x0C45,
  0x0C49,
  ...range(0x0C4E, 0x0C54),
  ...range(0x0C57, 0x0C5F),
  ...range(0x0C62, 0x0C65),
  ...range(0x0C70, 0x0C81),
  0x0C84,
  0x0C8D,
  0x0C91,
  0x0CA9,
  0x0CB4,
  ...range(0x0CBA, 0x0CBD),
  0x0CC5,
  0x0CC9,
  ...range(0x0CCE, 0x0CD4),
  ...range(0x0CD7, 0x0CDD),
  0x0CDF,
  ...range(0x0CE2, 0x0CE5),
  ...range(0x0CF0, 0x0D01),
  0x0D04,
  0x0D0D,
  0x0D11,
  0x0D29,
  ...range(0x0D3A, 0x0D3D),
  ...range(0x0D44, 0x0D45),
  0x0D49,
  ...range(0x0D4E, 0x0D56),
  ...range(0x0D58, 0x0D5F),
  ...range(0x0D62, 0x0D65),
  ...range(0x0D70, 0x0D81),
  0x0D84,
  ...range(0x0D97, 0x0D99),
  0x0DB2,
  0x0DBC,
  ...range(0x0DBE, 0x0DBF),
  ...range(0x0DC7, 0x0DC9),
  ...range(0x0DCB, 0x0DCE),
  0x0DD5,
  0x0DD7,
  ...range(0x0DE0, 0x0DF1),
  ...range(0x0DF5, 0x0E00),
  ...range(0x0E3B, 0x0E3E),
  ...range(0x0E5C, 0x0E80),
  0x0E83,
  ...range(0x0E85, 0x0E86),
  0x0E89,
  ...range(0x0E8B, 0x0E8C),
  ...range(0x0E8E, 0x0E93),
  0x0E98,
  0x0EA0,
  0x0EA4,
  0x0EA6,
  ...range(0x0EA8, 0x0EA9),
  0x0EAC,
  0x0EBA,
  ...range(0x0EBE, 0x0EBF),
  0x0EC5,
  0x0EC7,
  ...range(0x0ECE, 0x0ECF),
  ...range(0x0EDA, 0x0EDB),
  ...range(0x0EDE, 0x0EFF),
  0x0F48,
  ...range(0x0F6B, 0x0F70),
  ...range(0x0F8C, 0x0F8F),
  0x0F98,
  0x0FBD,
  ...range(0x0FCD, 0x0FCE),
  ...range(0x0FD0, 0x0FFF),
  0x1022,
  0x1028,
  0x102B,
  ...range(0x1033, 0x1035),
  ...range(0x103A, 0x103F),
  ...range(0x105A, 0x109F),
  ...range(0x10C6, 0x10CF),
  ...range(0x10F9, 0x10FA),
  ...range(0x10FC, 0x10FF),
  ...range(0x115A, 0x115E),
  ...range(0x11A3, 0x11A7),
  ...range(0x11FA, 0x11FF),
  0x1207,
  0x1247,
  0x1249,
  ...range(0x124E, 0x124F),
  0x1257,
  0x1259,
  ...range(0x125E, 0x125F),
  0x1287,
  0x1289,
  ...range(0x128E, 0x128F),
  0x12AF,
  0x12B1,
  ...range(0x12B6, 0x12B7),
  0x12BF,
  0x12C1,
  ...range(0x12C6, 0x12C7),
  0x12CF,
  0x12D7,
  0x12EF,
  0x130F,
  0x1311,
  ...range(0x1316, 0x1317),
  0x131F,
  0x1347,
  ...range(0x135B, 0x1360),
  ...range(0x137D, 0x139F),
  ...range(0x13F5, 0x1400),
  ...range(0x1677, 0x167F),
  ...range(0x169D, 0x169F),
  ...range(0x16F1, 0x16FF),
  0x170D,
  ...range(0x1715, 0x171F),
  ...range(0x1737, 0x173F),
  ...range(0x1754, 0x175F),
  0x176D,
  0x1771,
  ...range(0x1774, 0x177F),
  ...range(0x17DD, 0x17DF),
  ...range(0x17EA, 0x17FF),
  0x180F,
  ...range(0x181A, 0x181F),
  ...range(0x1878, 0x187F),
  ...range(0x18AA, 0x1DFF),
  ...range(0x1E9C, 0x1E9F),
  ...range(0x1EFA, 0x1EFF),
  ...range(0x1F16, 0x1F17),
  ...range(0x1F1E, 0x1F1F),
  ...range(0x1F46, 0x1F47),
  ...range(0x1F4E, 0x1F4F),
  0x1F58,
  0x1F5A,
  0x1F5C,
  0x1F5E,
  ...range(0x1F7E, 0x1F7F),
  0x1FB5,
  0x1FC5,
  ...range(0x1FD4, 0x1FD5),
  0x1FDC,
  ...range(0x1FF0, 0x1FF1),
  0x1FF5,
  0x1FFF,
  ...range(0x2053, 0x2056),
  ...range(0x2058, 0x205E),
  ...range(0x2064, 0x2069),
  ...range(0x2072, 0x2073),
  ...range(0x208F, 0x209F),
  ...range(0x20B2, 0x20CF),
  ...range(0x20EB, 0x20FF),
  ...range(0x213B, 0x213C),
  ...range(0x214C, 0x2152),
  ...range(0x2184, 0x218F),
  ...range(0x23CF, 0x23FF),
  ...range(0x2427, 0x243F),
  ...range(0x244B, 0x245F),
  0x24FF,
  ...range(0x2614, 0x2615),
  0x2618,
  ...range(0x267E, 0x267F),
  ...range(0x268A, 0x2700),
  0x2705,
  ...range(0x270A, 0x270B),
  0x2728,
  0x274C,
  0x274E,
  ...range(0x2753, 0x2755),
  0x2757,
  ...range(0x275F, 0x2760),
  ...range(0x2795, 0x2797),
  0x27B0,
  ...range(0x27BF, 0x27CF),
  ...range(0x27EC, 0x27EF),
  ...range(0x2B00, 0x2E7F),
  0x2E9A,
  ...range(0x2EF4, 0x2EFF),
  ...range(0x2FD6, 0x2FEF),
  ...range(0x2FFC, 0x2FFF),
  0x3040,
  ...range(0x3097, 0x3098),
  ...range(0x3100, 0x3104),
  ...range(0x312D, 0x3130),
  0x318F,
  ...range(0x31B8, 0x31EF),
  ...range(0x321D, 0x321F),
  ...range(0x3244, 0x3250),
  ...range(0x327C, 0x327E),
  ...range(0x32CC, 0x32CF),
  0x32FF,
  ...range(0x3377, 0x337A),
  ...range(0x33DE, 0x33DF),
  0x33FF,
  ...range(0x4DB6, 0x4DFF),
  ...range(0x9FA6, 0x9FFF),
  ...range(0xA48D, 0xA48F),
  ...range(0xA4C7, 0xABFF),
  ...range(0xD7A4, 0xD7FF),
  ...range(0xFA2E, 0xFA2F),
  ...range(0xFA6B, 0xFAFF),
  ...range(0xFB07, 0xFB12),
  ...range(0xFB18, 0xFB1C),
  0xFB37,
  0xFB3D,
  0xFB3F,
  0xFB42,
  0xFB45,
  ...range(0xFBB2, 0xFBD2),
  ...range(0xFD40, 0xFD4F),
  ...range(0xFD90, 0xFD91),
  ...range(0xFDC8, 0xFDCF),
  ...range(0xFDFD, 0xFDFF),
  ...range(0xFE10, 0xFE1F),
  ...range(0xFE24, 0xFE2F),
  ...range(0xFE47, 0xFE48),
  0xFE53,
  0xFE67,
  ...range(0xFE6C, 0xFE6F),
  0xFE75,
  ...range(0xFEFD, 0xFEFE),
  0xFF00,
  ...range(0xFFBF, 0xFFC1),
  ...range(0xFFC8, 0xFFC9),
  ...range(0xFFD0, 0xFFD1),
  ...range(0xFFD8, 0xFFD9),
  ...range(0xFFDD, 0xFFDF),
  0xFFE7,
  ...range(0xFFEF, 0xFFF8),
  ...range(0x10000, 0x102FF),
  0x1031F,
  ...range(0x10324, 0x1032F),
  ...range(0x1034B, 0x103FF),
  ...range(0x10426, 0x10427),
  ...range(0x1044E, 0x1CFFF),
  ...range(0x1D0F6, 0x1D0FF),
  ...range(0x1D127, 0x1D129),
  ...range(0x1D1DE, 0x1D3FF),
  0x1D455,
  0x1D49D,
  ...range(0x1D4A0, 0x1D4A1),
  ...range(0x1D4A3, 0x1D4A4),
  ...range(0x1D4A7, 0x1D4A8),
  0x1D4AD,
  0x1D4BA,
  0x1D4BC,
  0x1D4C1,
  0x1D4C4,
  0x1D506,
  ...range(0x1D50B, 0x1D50C),
  0x1D515,
  0x1D51D,
  0x1D53A,
  0x1D53F,
  0x1D545,
  ...range(0x1D547, 0x1D549),
  0x1D551,
  ...range(0x1D6A4, 0x1D6A7),
  ...range(0x1D7CA, 0x1D7CD),
  ...range(0x1D800, 0x1FFFD),
  ...range(0x2A6D7, 0x2F7FF),
  ...range(0x2FA1E, 0x2FFFD),
  ...range(0x30000, 0x3FFFD),
  ...range(0x40000, 0x4FFFD),
  ...range(0x50000, 0x5FFFD),
  ...range(0x60000, 0x6FFFD),
  ...range(0x70000, 0x7FFFD),
  ...range(0x80000, 0x8FFFD),
  ...range(0x90000, 0x9FFFD),
  ...range(0xA0000, 0xAFFFD),
  ...range(0xB0000, 0xBFFFD),
  ...range(0xC0000, 0xCFFFD),
  ...range(0xD0000, 0xDFFFD),
  0xE0000,
  ...range(0xE0002, 0xE001F),
  ...range(0xE0080, 0xEFFFD),
])

/**
 * B.1 Commonly mapped to nothing
 * @link https://tools.ietf.org/html/rfc3454#appendix-B.1
 */
const commonly_mapped_to_nothing = new Set([
  0x00AD,
  0x034F,
  0x1806,
  0x180B,
  0x180C,
  0x180D,
  0x200B,
  0x200C,
  0x200D,
  0x2060,
  0xFE00,
  0xFE01,
  0xFE02,
  0xFE03,
  0xFE04,
  0xFE05,
  0xFE06,
  0xFE07,
  0xFE08,
  0xFE09,
  0xFE0A,
  0xFE0B,
  0xFE0C,
  0xFE0D,
  0xFE0E,
  0xFE0F,
  0xFEFF,
])

/**
 * C.1.2 Non-ASCII space characters
 * @link https://tools.ietf.org/html/rfc3454#appendix-C.1.2
 */
const non_ASCII_space_characters = new Set([
  0x00a0 /* NO-BREAK SPACE */,
  0x1680 /* OGHAM SPACE MARK */,
  0x2000 /* EN QUAD */,
  0x2001 /* EM QUAD */,
  0x2002 /* EN SPACE */,
  0x2003 /* EM SPACE */,
  0x2004 /* THREE-PER-EM SPACE */,
  0x2005 /* FOUR-PER-EM SPACE */,
  0x2006 /* SIX-PER-EM SPACE */,
  0x2007 /* FIGURE SPACE */,
  0x2008 /* PUNCTUATION SPACE */,
  0x2009 /* THIN SPACE */,
  0x200a /* HAIR SPACE */,
  0x200b /* ZERO WIDTH SPACE */,
  0x202f /* NARROW NO-BREAK SPACE */,
  0x205f /* MEDIUM MATHEMATICAL SPACE */,
  0x3000 /* IDEOGRAPHIC SPACE */,
])

/**
 * 2.3.  Prohibited Output
 * @type {Set}
 */
const prohibited_characters = new Set([
  ...non_ASCII_space_characters,

  /**
   * C.2.1 ASCII control characters
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.2.1
   */
  ...range(0, 0x001f) /* [CONTROL CHARACTERS] */,
  0x007f /* DELETE */,

  /**
   * C.2.2 Non-ASCII control characters
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.2.2
   */
  ...range(0x0080, 0x009F), /* [CONTROL CHARACTERS] */
  0x06DD, /* ARABIC END OF AYAH */
  0x070F, /* SYRIAC ABBREVIATION MARK */
  0x180E, /* MONGOLIAN VOWEL SEPARATOR */
  0x200C, /* ZERO WIDTH NON-JOINER */
  0x200D, /* ZERO WIDTH JOINER */
  0x2028, /* LINE SEPARATOR */
  0x2029, /* PARAGRAPH SEPARATOR */
  0x2060, /* WORD JOINER */
  0x2061, /* FUNCTION APPLICATION */
  0x2062, /* INVISIBLE TIMES */
  0x2063, /* INVISIBLE SEPARATOR */
  ...range(0x206A, 0x206F), /* [CONTROL CHARACTERS] */
  0xFEFF, /* ZERO WIDTH NO-BREAK SPACE */
  ...range(0xFFF9, 0xFFFC), /* [CONTROL CHARACTERS] */
  ...range(0x1D173, 0x1D17A), /* [MUSICAL CONTROL CHARACTERS] */

  /**
   * C.3 Private use
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.3
   */
  ...range(0xE000, 0xF8FF), /* [PRIVATE USE, PLANE 0] */
  ...range(0xF0000, 0xFFFFD), /* [PRIVATE USE, PLANE 15] */
  ...range(0x100000, 0x10FFFD), /* [PRIVATE USE, PLANE 16] */

  /**
   * C.4 Non-character code points
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.4
   */
  ...range(0xFDD0, 0xFDEF), /* [NONCHARACTER CODE POINTS] */
  ...range(0xFFFE, 0xFFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x1FFFE, 0x1FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x2FFFE, 0x2FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x3FFFE, 0x3FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x4FFFE, 0x4FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x5FFFE, 0x5FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x6FFFE, 0x6FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x7FFFE, 0x7FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x8FFFE, 0x8FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x9FFFE, 0x9FFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0xAFFFE, 0xAFFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0xBFFFE, 0xBFFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0xCFFFE, 0xCFFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0xDFFFE, 0xDFFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0xEFFFE, 0xEFFFF), /* [NONCHARACTER CODE POINTS] */
  ...range(0x10FFFE, 0x10FFFF), /* [NONCHARACTER CODE POINTS] */

  /**
   * C.5 Surrogate codes
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.5
   */
  ...range(0xD800, 0xDFFF),

  /**
   * C.6 Inappropriate for plain text
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.6
   */
  0xFFF9, /* INTERLINEAR ANNOTATION ANCHOR */
  0xFFFA, /* INTERLINEAR ANNOTATION SEPARATOR */
  0xFFFB, /* INTERLINEAR ANNOTATION TERMINATOR */
  0xFFFC, /* OBJECT REPLACEMENT CHARACTER */
  0xFFFD, /* REPLACEMENT CHARACTER */

  /**
   * C.7 Inappropriate for canonical representation
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.7
   */
  ...range(0x2FF0, 0x2FFB), /* [IDEOGRAPHIC DESCRIPTION CHARACTERS] */

  /**
   * C.8 Change display properties or are deprecated
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.8
   */
  0x0340, /* COMBINING GRAVE TONE MARK */
  0x0341, /* COMBINING ACUTE TONE MARK */
  0x200E, /* LEFT-TO-RIGHT MARK */
  0x200F, /* RIGHT-TO-LEFT MARK */
  0x202A, /* LEFT-TO-RIGHT EMBEDDING */
  0x202B, /* RIGHT-TO-LEFT EMBEDDING */
  0x202C, /* POP DIRECTIONAL FORMATTING */
  0x202D, /* LEFT-TO-RIGHT OVERRIDE */
  0x202E, /* RIGHT-TO-LEFT OVERRIDE */
  0x206A, /* INHIBIT SYMMETRIC SWAPPING */
  0x206B, /* ACTIVATE SYMMETRIC SWAPPING */
  0x206C, /* INHIBIT ARABIC FORM SHAPING */
  0x206D, /* ACTIVATE ARABIC FORM SHAPING */
  0x206E, /* NATIONAL DIGIT SHAPES */
  0x206F, /* NOMINAL DIGIT SHAPES */

  /**
   * C.9 Tagging characters
   * @link https://tools.ietf.org/html/rfc3454#appendix-C.9
   */
  0xE0001, /* LANGUAGE TAG */
  ...range(0xE0020, 0xE007F) /* [TAGGING CHARACTERS] */
])

/**
 * D.1 Characters with bidirectional property "R" or "AL"
 * @link https://tools.ietf.org/html/rfc3454#appendix-D.1
 */
const bidirectional_r_al = new Set([
  0x05BE,
  0x05C0,
  0x05C3,
  ...range(0x05D0, 0x05EA),
  ...range(0x05F0, 0x05F4),
  0x061B,
  0x061F,
  ...range(0x0621, 0x063A),
  ...range(0x0640, 0x064A),
  ...range(0x066D, 0x066F),
  ...range(0x0671, 0x06D5),
  0x06DD,
  ...range(0x06E5, 0x06E6),
  ...range(0x06FA, 0x06FE),
  ...range(0x0700, 0x070D),
  0x0710,
  ...range(0x0712, 0x072C),
  ...range(0x0780, 0x07A5),
  0x07B1,
  0x200F,
  0xFB1D,
  ...range(0xFB1F, 0xFB28),
  ...range(0xFB2A, 0xFB36),
  ...range(0xFB38, 0xFB3C),
  0xFB3E,
  ...range(0xFB40, 0xFB41),
  ...range(0xFB43, 0xFB44),
  ...range(0xFB46, 0xFBB1),
  ...range(0xFBD3, 0xFD3D),
  ...range(0xFD50, 0xFD8F),
  ...range(0xFD92, 0xFDC7),
  ...range(0xFDF0, 0xFDFC),
  ...range(0xFE70, 0xFE74),
  ...range(0xFE76, 0xFEFC),
])

/**
 * D.2 Characters with bidirectional property "L"
 * @link https://tools.ietf.org/html/rfc3454#appendix-D.2
 */
const bidirectional_l = new Set([
  ...range(0x0041, 0x005A),
  ...range(0x0061, 0x007A),
  0x00AA,
  0x00B5,
  0x00BA,
  ...range(0x00C0, 0x00D6),
  ...range(0x00D8, 0x00F6),
  ...range(0x00F8, 0x0220),
  ...range(0x0222, 0x0233),
  ...range(0x0250, 0x02AD),
  ...range(0x02B0, 0x02B8),
  ...range(0x02BB, 0x02C1),
  ...range(0x02D0, 0x02D1),
  ...range(0x02E0, 0x02E4),
  0x02EE,
  0x037A,
  0x0386,
  ...range(0x0388, 0x038A),
  0x038C,
  ...range(0x038E, 0x03A1),
  ...range(0x03A3, 0x03CE),
  ...range(0x03D0, 0x03F5),
  ...range(0x0400, 0x0482),
  ...range(0x048A, 0x04CE),
  ...range(0x04D0, 0x04F5),
  ...range(0x04F8, 0x04F9),
  ...range(0x0500, 0x050F),
  ...range(0x0531, 0x0556),
  ...range(0x0559, 0x055F),
  ...range(0x0561, 0x0587),
  0x0589,
  0x0903,
  ...range(0x0905, 0x0939),
  ...range(0x093D, 0x0940),
  ...range(0x0949, 0x094C),
  0x0950,
  ...range(0x0958, 0x0961),
  ...range(0x0964, 0x0970),
  ...range(0x0982, 0x0983),
  ...range(0x0985, 0x098C),
  ...range(0x098F, 0x0990),
  ...range(0x0993, 0x09A8),
  ...range(0x09AA, 0x09B0),
  0x09B2,
  ...range(0x09B6, 0x09B9),
  ...range(0x09BE, 0x09C0),
  ...range(0x09C7, 0x09C8),
  ...range(0x09CB, 0x09CC),
  0x09D7,
  ...range(0x09DC, 0x09DD),
  ...range(0x09DF, 0x09E1),
  ...range(0x09E6, 0x09F1),
  ...range(0x09F4, 0x09FA),
  ...range(0x0A05, 0x0A0A),
  ...range(0x0A0F, 0x0A10),
  ...range(0x0A13, 0x0A28),
  ...range(0x0A2A, 0x0A30),
  ...range(0x0A32, 0x0A33),
  ...range(0x0A35, 0x0A36),
  ...range(0x0A38, 0x0A39),
  ...range(0x0A3E, 0x0A40),
  ...range(0x0A59, 0x0A5C),
  0x0A5E,
  ...range(0x0A66, 0x0A6F),
  ...range(0x0A72, 0x0A74),
  0x0A83,
  ...range(0x0A85, 0x0A8B),
  0x0A8D,
  ...range(0x0A8F, 0x0A91),
  ...range(0x0A93, 0x0AA8),
  ...range(0x0AAA, 0x0AB0),
  ...range(0x0AB2, 0x0AB3),
  ...range(0x0AB5, 0x0AB9),
  ...range(0x0ABD, 0x0AC0),
  0x0AC9,
  ...range(0x0ACB, 0x0ACC),
  0x0AD0,
  0x0AE0,
  ...range(0x0AE6, 0x0AEF),
  ...range(0x0B02, 0x0B03),
  ...range(0x0B05, 0x0B0C),
  ...range(0x0B0F, 0x0B10),
  ...range(0x0B13, 0x0B28),
  ...range(0x0B2A, 0x0B30),
  ...range(0x0B32, 0x0B33),
  ...range(0x0B36, 0x0B39),
  ...range(0x0B3D, 0x0B3E),
  0x0B40,
  ...range(0x0B47, 0x0B48),
  ...range(0x0B4B, 0x0B4C),
  0x0B57,
  ...range(0x0B5C, 0x0B5D),
  ...range(0x0B5F, 0x0B61),
  ...range(0x0B66, 0x0B70),
  0x0B83,
  ...range(0x0B85, 0x0B8A),
  ...range(0x0B8E, 0x0B90),
  ...range(0x0B92, 0x0B95),
  ...range(0x0B99, 0x0B9A),
  0x0B9C,
  ...range(0x0B9E, 0x0B9F),
  ...range(0x0BA3, 0x0BA4),
  ...range(0x0BA8, 0x0BAA),
  ...range(0x0BAE, 0x0BB5),
  ...range(0x0BB7, 0x0BB9),
  ...range(0x0BBE, 0x0BBF),
  ...range(0x0BC1, 0x0BC2),
  ...range(0x0BC6, 0x0BC8),
  ...range(0x0BCA, 0x0BCC),
  0x0BD7,
  ...range(0x0BE7, 0x0BF2),
  ...range(0x0C01, 0x0C03),
  ...range(0x0C05, 0x0C0C),
  ...range(0x0C0E, 0x0C10),
  ...range(0x0C12, 0x0C28),
  ...range(0x0C2A, 0x0C33),
  ...range(0x0C35, 0x0C39),
  ...range(0x0C41, 0x0C44),
  ...range(0x0C60, 0x0C61),
  ...range(0x0C66, 0x0C6F),
  ...range(0x0C82, 0x0C83),
  ...range(0x0C85, 0x0C8C),
  ...range(0x0C8E, 0x0C90),
  ...range(0x0C92, 0x0CA8),
  ...range(0x0CAA, 0x0CB3),
  ...range(0x0CB5, 0x0CB9),
  0x0CBE,
  ...range(0x0CC0, 0x0CC4),
  ...range(0x0CC7, 0x0CC8),
  ...range(0x0CCA, 0x0CCB),
  ...range(0x0CD5, 0x0CD6),
  0x0CDE,
  ...range(0x0CE0, 0x0CE1),
  ...range(0x0CE6, 0x0CEF),
  ...range(0x0D02, 0x0D03),
  ...range(0x0D05, 0x0D0C),
  ...range(0x0D0E, 0x0D10),
  ...range(0x0D12, 0x0D28),
  ...range(0x0D2A, 0x0D39),
  ...range(0x0D3E, 0x0D40),
  ...range(0x0D46, 0x0D48),
  ...range(0x0D4A, 0x0D4C),
  0x0D57,
  ...range(0x0D60, 0x0D61),
  ...range(0x0D66, 0x0D6F),
  ...range(0x0D82, 0x0D83),
  ...range(0x0D85, 0x0D96),
  ...range(0x0D9A, 0x0DB1),
  ...range(0x0DB3, 0x0DBB),
  0x0DBD,
  ...range(0x0DC0, 0x0DC6),
  ...range(0x0DCF, 0x0DD1),
  ...range(0x0DD8, 0x0DDF),
  ...range(0x0DF2, 0x0DF4),
  ...range(0x0E01, 0x0E30),
  ...range(0x0E32, 0x0E33),
  ...range(0x0E40, 0x0E46),
  ...range(0x0E4F, 0x0E5B),
  ...range(0x0E81, 0x0E82),
  0x0E84,
  ...range(0x0E87, 0x0E88),
  0x0E8A,
  0x0E8D,
  ...range(0x0E94, 0x0E97),
  ...range(0x0E99, 0x0E9F),
  ...range(0x0EA1, 0x0EA3),
  0x0EA5,
  0x0EA7,
  ...range(0x0EAA, 0x0EAB),
  ...range(0x0EAD, 0x0EB0),
  ...range(0x0EB2, 0x0EB3),
  0x0EBD,
  ...range(0x0EC0, 0x0EC4),
  0x0EC6,
  ...range(0x0ED0, 0x0ED9),
  ...range(0x0EDC, 0x0EDD),
  ...range(0x0F00, 0x0F17),
  ...range(0x0F1A, 0x0F34),
  0x0F36,
  0x0F38,
  ...range(0x0F3E, 0x0F47),
  ...range(0x0F49, 0x0F6A),
  0x0F7F,
  0x0F85,
  ...range(0x0F88, 0x0F8B),
  ...range(0x0FBE, 0x0FC5),
  ...range(0x0FC7, 0x0FCC),
  0x0FCF,
  ...range(0x1000, 0x1021),
  ...range(0x1023, 0x1027),
  ...range(0x1029, 0x102A),
  0x102C,
  0x1031,
  0x1038,
  ...range(0x1040, 0x1057),
  ...range(0x10A0, 0x10C5),
  ...range(0x10D0, 0x10F8),
  0x10FB,
  ...range(0x1100, 0x1159),
  ...range(0x115F, 0x11A2),
  ...range(0x11A8, 0x11F9),
  ...range(0x1200, 0x1206),
  ...range(0x1208, 0x1246),
  0x1248,
  ...range(0x124A, 0x124D),
  ...range(0x1250, 0x1256),
  0x1258,
  ...range(0x125A, 0x125D),
  ...range(0x1260, 0x1286),
  0x1288,
  ...range(0x128A, 0x128D),
  ...range(0x1290, 0x12AE),
  0x12B0,
  ...range(0x12B2, 0x12B5),
  ...range(0x12B8, 0x12BE),
  0x12C0,
  ...range(0x12C2, 0x12C5),
  ...range(0x12C8, 0x12CE),
  ...range(0x12D0, 0x12D6),
  ...range(0x12D8, 0x12EE),
  ...range(0x12F0, 0x130E),
  0x1310,
  ...range(0x1312, 0x1315),
  ...range(0x1318, 0x131E),
  ...range(0x1320, 0x1346),
  ...range(0x1348, 0x135A),
  ...range(0x1361, 0x137C),
  ...range(0x13A0, 0x13F4),
  ...range(0x1401, 0x1676),
  ...range(0x1681, 0x169A),
  ...range(0x16A0, 0x16F0),
  ...range(0x1700, 0x170C),
  ...range(0x170E, 0x1711),
  ...range(0x1720, 0x1731),
  ...range(0x1735, 0x1736),
  ...range(0x1740, 0x1751),
  ...range(0x1760, 0x176C),
  ...range(0x176E, 0x1770),
  ...range(0x1780, 0x17B6),
  ...range(0x17BE, 0x17C5),
  ...range(0x17C7, 0x17C8),
  ...range(0x17D4, 0x17DA),
  0x17DC,
  ...range(0x17E0, 0x17E9),
  ...range(0x1810, 0x1819),
  ...range(0x1820, 0x1877),
  ...range(0x1880, 0x18A8),
  ...range(0x1E00, 0x1E9B),
  ...range(0x1EA0, 0x1EF9),
  ...range(0x1F00, 0x1F15),
  ...range(0x1F18, 0x1F1D),
  ...range(0x1F20, 0x1F45),
  ...range(0x1F48, 0x1F4D),
  ...range(0x1F50, 0x1F57),
  0x1F59,
  0x1F5B,
  0x1F5D,
  ...range(0x1F5F, 0x1F7D),
  ...range(0x1F80, 0x1FB4),
  ...range(0x1FB6, 0x1FBC),
  0x1FBE,
  ...range(0x1FC2, 0x1FC4),
  ...range(0x1FC6, 0x1FCC),
  ...range(0x1FD0, 0x1FD3),
  ...range(0x1FD6, 0x1FDB),
  ...range(0x1FE0, 0x1FEC),
  ...range(0x1FF2, 0x1FF4),
  ...range(0x1FF6, 0x1FFC),
  0x200E,
  0x2071,
  0x207F,
  0x2102,
  0x2107,
  ...range(0x210A, 0x2113),
  0x2115,
  ...range(0x2119, 0x211D),
  0x2124,
  0x2126,
  0x2128,
  ...range(0x212A, 0x212D),
  ...range(0x212F, 0x2131),
  ...range(0x2133, 0x2139),
  ...range(0x213D, 0x213F),
  ...range(0x2145, 0x2149),
  ...range(0x2160, 0x2183),
  ...range(0x2336, 0x237A),
  0x2395,
  ...range(0x249C, 0x24E9),
  ...range(0x3005, 0x3007),
  ...range(0x3021, 0x3029),
  ...range(0x3031, 0x3035),
  ...range(0x3038, 0x303C),
  ...range(0x3041, 0x3096),
  ...range(0x309D, 0x309F),
  ...range(0x30A1, 0x30FA),
  ...range(0x30FC, 0x30FF),
  ...range(0x3105, 0x312C),
  ...range(0x3131, 0x318E),
  ...range(0x3190, 0x31B7),
  ...range(0x31F0, 0x321C),
  ...range(0x3220, 0x3243),
  ...range(0x3260, 0x327B),
  ...range(0x327F, 0x32B0),
  ...range(0x32C0, 0x32CB),
  ...range(0x32D0, 0x32FE),
  ...range(0x3300, 0x3376),
  ...range(0x337B, 0x33DD),
  ...range(0x33E0, 0x33FE),
  ...range(0x3400, 0x4DB5),
  ...range(0x4E00, 0x9FA5),
  ...range(0xA000, 0xA48C),
  ...range(0xAC00, 0xD7A3),
  ...range(0xD800, 0xFA2D),
  ...range(0xFA30, 0xFA6A),
  ...range(0xFB00, 0xFB06),
  ...range(0xFB13, 0xFB17),
  ...range(0xFF21, 0xFF3A),
  ...range(0xFF41, 0xFF5A),
  ...range(0xFF66, 0xFFBE),
  ...range(0xFFC2, 0xFFC7),
  ...range(0xFFCA, 0xFFCF),
  ...range(0xFFD2, 0xFFD7),
  ...range(0xFFDA, 0xFFDC),
  ...range(0x10300, 0x1031E),
  ...range(0x10320, 0x10323),
  ...range(0x10330, 0x1034A),
  ...range(0x10400, 0x10425),
  ...range(0x10428, 0x1044D),
  ...range(0x1D000, 0x1D0F5),
  ...range(0x1D100, 0x1D126),
  ...range(0x1D12A, 0x1D166),
  ...range(0x1D16A, 0x1D172),
  ...range(0x1D183, 0x1D184),
  ...range(0x1D18C, 0x1D1A9),
  ...range(0x1D1AE, 0x1D1DD),
  ...range(0x1D400, 0x1D454),
  ...range(0x1D456, 0x1D49C),
  ...range(0x1D49E, 0x1D49F),
  0x1D4A2,
  ...range(0x1D4A5, 0x1D4A6),
  ...range(0x1D4A9, 0x1D4AC),
  ...range(0x1D4AE, 0x1D4B9),
  0x1D4BB,
  ...range(0x1D4BD, 0x1D4C0),
  ...range(0x1D4C2, 0x1D4C3),
  ...range(0x1D4C5, 0x1D505),
  ...range(0x1D507, 0x1D50A),
  ...range(0x1D50D, 0x1D514),
  ...range(0x1D516, 0x1D51C),
  ...range(0x1D51E, 0x1D539),
  ...range(0x1D53B, 0x1D53E),
  ...range(0x1D540, 0x1D544),
  0x1D546,
  ...range(0x1D54A, 0x1D550),
  ...range(0x1D552, 0x1D6A3),
  ...range(0x1D6A8, 0x1D7C9),
  ...range(0x20000, 0x2A6D6),
  ...range(0x2F800, 0x2FA1D),
  ...range(0xF0000, 0xFFFFD),
  ...range(0x100000, 0x10FFFD)
])

module.exports = {
  unassigned_code_points,
  commonly_mapped_to_nothing,
  non_ASCII_space_characters,
  prohibited_characters,
  bidirectional_r_al,
  bidirectional_l
}
