import {EventEmitter} from 'events'

type Callback<T> = (err: Error, val: T) => any

/**
 * 
 */
/**
 * Queryable represents any object that can execute database queries. 
 *
 * This is the most commonly used interface in `any-db` and is implemented
 * by `Connection`, `Transaction` and `ConnectionPool`.
 */
export interface Queryable<Params> extends EventEmitter {
    /**
     * The Adapter instance that will be used by this Queryable for creating
     * Query instances and/or connections.
     */
    adapter: Adapter<Params>

    /**
     * Execute a SQL statement using bound parameters (if they are provided)
     * and return a Query object that is a Readable stream of the resulting
     * rows. If a `Callback<ResultSet>` is provided the rows returned by the
     * database will be aggregated into a `ResultSet` which will be passed to
     * the continuation after the query has completed.
     *
     * Example of collecting results by passing a callback:
     * 
     *     queryable.query('SELECT * FROM my_table', function (err, res) {
     *       if (err) return console.error(err)
     *       res.rows.forEach(console.log)
     *       console.log('All done!')
     *     })
     *
     * Example of streaming query results:
     * 
     *     queryable.query('SELECT * FROM my_table')
     *       .on('error', console.error)
     *       .on('data', console.log)
     *       .on('end', function () { console.log('All done!') })
     */
    query<Row>(text: String, params?: Params, callback?: Callback<ResultSet<Row>>): Query<Row>

    /**
     * Execute a pre-constructed `Query`
     * 
     * This is not needed for normal application use, but is used by support
     * libraries such as `any-db-pool` and `any-db-transaction`.
     * 
     * @see Adapter.createQuery for more details.
     */
    query<Row>(query: Query<Row>): Query<Row>

    /**
     * The `'query'` event is emitted immediately before a `Query` is executed.
     *
     * You can use this event to implement query logging/metrics.
     */
    on(event: 'query', listener: (query: Query<Params, any>) => any): this
    once(event: 'query', listener: (query: Query<Params, any>) => any): this
}

export interface Connection<Params> extends Queryable<Params> {
    /**
     * The `'query'` event is emitted immediately before a `Query` is executed.
     *
     * You can use this event to implement query logging/metrics.
     */
    on(event: 'query', listener: (query: Query<Params, any>) => any): this
    once(event: 'query', listener: (query: Query<Params, any>) => any): this

    /**
     * The `'error'` event is emitted when there is a connection-level error that
     * is not associated with a `Query`.
     */
    on(event: 'error', listener: (err: Error) => any): this
    once(event: 'error', listener: (err: Error) => any): this

    /**
     * The `'open'` event is emitted when the connection has been established and is ready to query.
     */
    on(event: 'open', listener: () => any): this
    once(event: 'open', listener: () => any): this

    /**
     * The `'close'` event is emitted when the connection has been closed.
     */
    on(event: 'close', listener: () => any): this
    once(event: 'close', listener: () => any): this
}

export interface Query<Params, Row> {
    text: string
    values: Params
    callback?: Callback<ResultSet<Row>>

    /*
     * The `'fields'` event is emitted once per query with driver-specific information about fields in the query results.
     */
    on(event: 'fields', listener: (fields: Field[]) => any): this
    once(event: 'fields', listener: (fields: Field[]) => any): this

    /*
     * The `'data'` event is emitted once per row in the query results.
     */
    on(event: 'data', listener: (row: Row) => any): this
    once(event: 'data', listener: (row: Row) => any): this

    /*
     * The `'close'` event is emitted once after all query results have been received.
     */
    on(event: 'close', listener: () => any): this
    once(event: 'close', listener: () => any): this

    /*
     * The `'end'` event is emitted once after all query results have been consumed.
     */
    on(event: 'end', listener: () => any): this
    once(event: 'end', listener: () => any): this
}

export interface Field {
    name: string
}

export interface ResultSet<Row> {
    fields: Field[]
    rows: Row[]
}

/**
 * An adapter is the object that provides a conforming interface for a particular database backend.
 */
export interface Adapter<Params> {
    /** The unique string name of this adapter. */
    name: string

    /**
     * Create a Query that _may_ eventually be executed by a `Connection`.
     * 
     * This function is rarely needed by user code, but enables the
     * `ConnectionPool` and `Transaction` types to return a `Query` object
     * before a connection is necessarily available/ready for use.
     */
    createQuery<Row>(text: string, params?: Params, callback?: Callback<ResultSet<Row>>): Query<Row>

    /**
     * Open a new database connection.
     */
    createConnection(config: {}, callback?: Callback<Connection<Params>>): Connection
}